/*
 * Decompiled with CFR 0.152.
 */
package org.sagebionetworks.bridge.android.manager;

import androidx.annotation.AnyThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteSink;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import javax.inject.Inject;
import javax.inject.Named;
import okhttp3.OkHttpClient;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.sagebionetworks.bridge.android.di.BridgeStudyParticipantScope;
import org.sagebionetworks.bridge.android.manager.AuthenticationManager;
import org.sagebionetworks.bridge.android.manager.BridgeManagerProvider;
import org.sagebionetworks.bridge.android.manager.dao.UploadDAO;
import org.sagebionetworks.bridge.android.manager.upload.FileUploadRequestBody;
import org.sagebionetworks.bridge.android.manager.upload.S3Service;
import org.sagebionetworks.bridge.android.util.retrofit.RxUtils;
import org.sagebionetworks.bridge.data.AndroidStudyUploadEncryptor;
import org.sagebionetworks.bridge.data.Archive;
import org.sagebionetworks.bridge.rest.model.UploadRequest;
import org.sagebionetworks.bridge.rest.model.UploadSession;
import org.sagebionetworks.bridge.rest.model.UploadValidationStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.cms.CMSException;
import retrofit2.Retrofit;
import rx.Completable;
import rx.Observable;
import rx.Single;
import rx.schedulers.Schedulers;

@BridgeStudyParticipantScope
@AnyThread
public class UploadManager
implements AuthenticationManager.AuthenticationEventListener {
    private static final Logger LOG = LoggerFactory.getLogger(UploadManager.class);
    private static final String CONTENT_TYPE_DATA_ARCHIVE = "application/zip";
    private static final int UPLOAD_EXPIRY_WINDOW_MINUTES = 30;
    private final AtomicReference<AuthenticationManager.AuthStateHolder> authenticatedSafeAtomicReference;
    private final AndroidStudyUploadEncryptor encryptor;
    private final UploadDAO uploadDAO;
    private final OkHttpClient s3OkHttpClient;

    @Inject
    public UploadManager(AuthenticationManager authenticationManager, AndroidStudyUploadEncryptor encryptor, UploadDAO uploadDAO, @Named(value="s3OkHttp3Client") OkHttpClient s3Okhttp3Client) {
        this.authenticatedSafeAtomicReference = authenticationManager.getAuthStateReference();
        authenticationManager.addEventListener(this);
        this.encryptor = encryptor;
        this.uploadDAO = uploadDAO;
        this.s3OkHttpClient = s3Okhttp3Client;
    }

    @NonNull
    public Single<UploadFile> queueUpload(String filename, Archive archive) {
        LOG.debug("Queueing archive: " + archive);
        return Single.fromCallable(() -> this.persist(filename, archive)).subscribeOn(Schedulers.io());
    }

    Completable clearUploads() {
        return Completable.merge((Observable)this.getUploadFilenames().map(this::dequeueUpload));
    }

    @Override
    public void onSignedOut(String email) {
        this.clearUploads().subscribe();
    }

    @Override
    public void onSignedIn(String email) {
    }

    @NonNull
    public Completable processUploadFiles() {
        return Completable.concat((Observable)this.getUploadFilenames().map(this.uploadDAO::getUploadFile).map(uploadFile -> this.processUploadForCachedSession((UploadFile)uploadFile, this.uploadDAO.getUploadSession(uploadFile.filename)).doOnError(t -> LOG.warn("Failed to process upload file: {}", uploadFile, t))).onErrorReturn(t -> Completable.complete()));
    }

    Observable<String> getUploadFilenames() {
        Set<String> filenames = this.uploadDAO.listUploadFilenames();
        return Observable.from((Object[])filenames.toArray(new String[filenames.size()]));
    }

    @NonNull
    public Completable processUploadFile(@NonNull UploadFile uploadFile) {
        Preconditions.checkNotNull((Object)uploadFile);
        Single cachedSessionSingle = Single.just((Object)this.uploadDAO.getUploadSession(uploadFile.filename));
        return Single.zip((Single)Single.just((Object)uploadFile), (Single)cachedSessionSingle, this::processUploadForCachedSession).flatMapCompletable(i -> i);
    }

    @NonNull
    Completable processUploadForCachedSession(@NonNull UploadFile uploadFile, @Nullable UploadSession cachedSession) {
        Preconditions.checkNotNull((Object)uploadFile, (Object)"uploadFile cannot be null");
        Single sessionSingle = cachedSession != null ? Single.just((Object)cachedSession) : this.getUploadSession(uploadFile).cache();
        Single statusSingle = sessionSingle.flatMap(uploadSession -> this.getUploadValidationStatus(uploadSession.getId()));
        return Single.zip((Single)Single.just((Object)uploadFile), (Single)sessionSingle, (Single)statusSingle, this::processUploadForValidationStatus).doOnError(t -> LOG.warn("Failed to process upload for cached session for file: {}", (Object)uploadFile.filename, t)).flatMapCompletable(i -> i);
    }

    @NonNull
    Completable processUploadForValidationStatus(@NonNull UploadFile uploadFile, @NonNull UploadSession uploadSession, @NonNull UploadValidationStatus uploadValidationStatus) {
        Preconditions.checkNotNull((Object)uploadFile, (Object)"uploadFile cannot be null");
        Preconditions.checkNotNull((Object)uploadSession, (Object)"uploadSession cannot be null");
        Preconditions.checkNotNull((Object)uploadValidationStatus, (Object)"uploadValidationSession cannot be null");
        switch (uploadValidationStatus.getStatus()) {
            case REQUESTED: {
                return this.uploadToS3(uploadFile, uploadSession);
            }
            case SUCCEEDED: 
            case DUPLICATE: {
                return this.dequeueUpload(uploadFile.filename);
            }
            case VALIDATION_IN_PROGRESS: {
                LOG.debug("Validation in progress for filename: " + uploadFile.filename + ", uploadId" + uploadSession.getId());
                break;
            }
            case VALIDATION_FAILED: {
                LOG.debug("Validation failed for filename: " + uploadFile.filename + ", uploadId" + uploadSession.getId());
                break;
            }
            default: {
                LOG.warn("Unknown status for uploadId: " + uploadValidationStatus.getId());
            }
        }
        return Completable.complete();
    }

    @NonNull
    Completable dequeueUpload(@NonNull String filename) {
        Preconditions.checkNotNull((Object)filename, (Object)"filename required");
        return Completable.fromAction(() -> {
            if (this.getFile(filename).delete()) {
                LOG.info("Successfully deleted upload file: " + filename + ", removing upload from queue");
            } else {
                LOG.warn("Failed to delete upload file: " + filename);
            }
        }).doOnCompleted(() -> this.uploadDAO.removeUploadAndSession(filename)).subscribeOn(Schedulers.io());
    }

    @NonNull
    Single<UploadValidationStatus> getUploadValidationStatus(@NonNull String uploadId) {
        Preconditions.checkNotNull((Object)uploadId, (Object)"uploadId required");
        return RxUtils.toBodySingle(this.authenticatedSafeAtomicReference.get().forConsentedUsersApi.getUploadStatus(uploadId)).doOnError(t -> LOG.warn("Failed to retrieve validation status for upload with id: {}", (Object)uploadId, t));
    }

    @NonNull
    Completable uploadToS3(UploadFile uploadFile, UploadSession session) {
        File file = this.getFile(uploadFile.filename);
        Preconditions.checkArgument((boolean)file.exists(), (Object)("Non-existent file: " + file.getAbsolutePath()));
        Single sessionSingle = Single.just((Object)session).flatMap(uploadSession -> {
            DateTime desiredMinimumExpiration = DateTime.now().plusMinutes(30);
            if (uploadSession.getExpires().isBefore((ReadableInstant)desiredMinimumExpiration)) {
                return this.getUploadSession(uploadFile);
            }
            return Single.just((Object)session);
        });
        LOG.info("Attempting S3 upload for file: {}, sessionId: {}", (Object)uploadFile.filename, (Object)session.getId());
        FileUploadRequestBody requestBody = new FileUploadRequestBody(file, uploadFile.contentType, l -> LOG.trace("File {}: Uploaded {} of {} bytes", new Object[]{uploadFile.filename, l, uploadFile.fileLength}));
        return sessionSingle.flatMap(freshSession -> RxUtils.toBodySingle(this.getS3Service((UploadSession)freshSession).uploadToS3(freshSession.getUrl(), requestBody, uploadFile.md5Hash, uploadFile.contentType))).doOnSuccess(aVoid -> {
            LOG.info("S3 upload succeeded for file: {}, sessionId: {}", (Object)uploadFile.filename, (Object)session.getId());
            RxUtils.toBodySingle(this.authenticatedSafeAtomicReference.get().forConsentedUsersApi.completeUploadSession(session.getId(), Boolean.valueOf(false), Boolean.valueOf(false))).doOnSuccess(val -> LOG.info("Call to upload complete succeeded")).onErrorReturn(t -> {
                LOG.info("Call to upload complete failed, server will recover", t);
                return null;
            }).subscribe();
        }).doOnError(t -> LOG.warn("S3 upload failed for file: {}, sessionId: {}", new Object[]{uploadFile.filename, session.getId(), t})).toCompletable();
    }

    @NonNull
    Single<UploadSession> getUploadSession(UploadFile uploadFile) {
        return RxUtils.toBodySingle(this.authenticatedSafeAtomicReference.get().forConsentedUsersApi.requestUploadSession(new UploadRequest().name(uploadFile.filename).contentType(uploadFile.contentType).contentLength(Long.valueOf(uploadFile.fileLength)).contentMd5(uploadFile.md5Hash))).doOnSuccess(uploadSession -> {
            LOG.info("Received processUploadFiles session with id: " + uploadSession.getId());
            this.uploadDAO.putUploadSession(uploadFile.filename, (UploadSession)uploadSession);
        }).doOnError(t -> LOG.warn("Failed to get upload session for file: " + uploadFile.filename, t));
    }

    @WorkerThread
    @Nullable
    UploadFile persist(String filename, Archive archive) throws IOException, CMSException, NoSuchAlgorithmException {
        MessageDigest md5;
        File file = this.getFile(filename);
        ByteSink sink = Files.asByteSink((File)file, (FileWriteMode[])new FileWriteMode[]{FileWriteMode.APPEND});
        try {
            md5 = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            LOG.error("MD5 digest not found", (Throwable)e);
            throw e;
        }
        LOG.debug("Writing archive with filename: " + filename + ", with contents: " + archive);
        try (OutputStream os = sink.openBufferedStream();
             DigestOutputStream md5OutStream = new DigestOutputStream(os, md5);
             OutputStream encryptedOutputStream = this.encryptor.encrypt((OutputStream)md5OutStream);){
            archive.writeTo(encryptedOutputStream);
        }
        catch (CMSException e) {
            LOG.warn("Failed to write archive with filename: {}", (Object)filename, (Object)e);
            return null;
        }
        String md5Hash = BaseEncoding.base64().encode(md5.digest());
        UploadFile uploadFile = new UploadFile();
        uploadFile.filename = filename;
        uploadFile.contentType = CONTENT_TYPE_DATA_ARCHIVE;
        uploadFile.fileLength = file.length();
        uploadFile.md5Hash = md5Hash;
        uploadFile.createdOn = DateTime.now();
        this.uploadDAO.putUploadFile(filename, uploadFile);
        return uploadFile;
    }

    File getFile(String filename) {
        return new File(BridgeManagerProvider.getInstance().getApplicationContext().getFilesDir().getAbsolutePath() + File.separator + filename);
    }

    S3Service getS3Service(UploadSession uploadSession) {
        URI uri = URI.create(uploadSession.getUrl());
        String baseUrl = uri.getScheme() + "://" + uri.getHost() + "/";
        Retrofit retrofit = new Retrofit.Builder().baseUrl(baseUrl).client(this.s3OkHttpClient).build();
        return (S3Service)retrofit.create(S3Service.class);
    }

    public static class UploadFile {
        public String filename;
        public String contentType;
        public long fileLength;
        public String md5Hash;
        public DateTime createdOn;
    }
}

