package org.findmykids.geo.domain.live.locator

import io.reactivex.Observable
import org.findmykids.geo._todo.Geo
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Lbs
import org.findmykids.geo.data.model.Location
import org.findmykids.geo.data.model.WifiFull
import org.findmykids.geo.data.repository.live.battery.BatteryRepository
import org.findmykids.geo.data.repository.live.lbs.LbsRepository
import org.findmykids.geo.data.repository.live.wifi.WifiRepository
import org.findmykids.geo.data.repository.storage.configuration.ConfigurationRepository
import org.findmykids.geo.data.repository.storage.currentSession.CurrentSessionRepository
import org.findmykids.geo.data.repository.storage.geo.GeoRepository
import org.findmykids.geo.data.repository.storage.yandexLocator.YandexLocatorRepository
import org.findmykids.geo.domain.BaseInteractor
import org.findmykids.geo.domain.model.InnerEvent
import java.util.*
import javax.inject.Inject


//TODO яндекс
internal class LocatorInteractorImpl @Inject constructor(
    private val mGeoRepository: GeoRepository,
    private val mConfigurationRepository: ConfigurationRepository,
    private val mCurrentSessionRepository: CurrentSessionRepository,
    private val mYandexLocatorRepository: YandexLocatorRepository,
    private val mBatteryRepository: BatteryRepository,
    private val mLbsRepository: LbsRepository,
    private val mWifiRepository: WifiRepository
) : BaseInteractor(),
    LocatorInteractor {

    private var mExtensions: MutableMap<String, String> = mutableMapOf()
    private var mLastSendedGeo: Geo? = null
    private var mNonSended = mutableListOf<Geo>()
    private var mLastLbsList = listOf<Lbs>()
    private var mLastWifiList = listOf<WifiFull>()


    override fun toString(): String = ""


    override fun update(): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .doOnSubscribe {
            mExtensions.clear()
        }
        .flatMapObservable {
            // подписка
            Logger.d().with(this@LocatorInteractorImpl).print()
            Observable.merge(
                batteryEmitter(it.batteryDataConfiguration),
                lbsEmitter(it.lbsDataConfiguration),
                wifiEmitter(it.wifiDataConfiguration)
            )
        }
        .concatMap { geo ->
            // получаем актуальную конфигурацию
            Logger.d().addArg(geo).with(this@LocatorInteractorImpl).print()
            mConfigurationRepository
                .get()
                .flatMapObservable { configurations ->
                    // в зависимоти от конфигурации обрабаываем гео
                    Logger.d().addArg(geo).addArg(configurations).with(this@LocatorInteractorImpl).print()
                    synchronized(mNonSended) {
                        when {
                            geo.session.isRealtime -> {
                                Logger.d().with(this@LocatorInteractorImpl).print()
                                mNonSended.clear()
                                mGeoRepository
                                    .sendGeo(geo)
                                    .andThen(Observable.just(InnerEvent.NewGeo(geo)))
                            }
                            isSendCriteria(geo, configurations.locatorLiveConfiguration) -> {
                                Logger.d().with(this@LocatorInteractorImpl).print()
                                mNonSended.clear()
                                mGeoRepository
                                    .sendGeo(geo)
                                    .andThen(Observable.just(InnerEvent.NewGeo(geo)))
                            }
                            else -> {
                                Logger.d().with(this@LocatorInteractorImpl).print()
                                mNonSended.add(geo)
                                Observable.just(InnerEvent.NewGeo(geo))
                            }
                        }
                    }
                }
        }


    private fun batteryEmitter(configuration: Configuration.BatteryDataConfiguration): Observable<Geo> = mBatteryRepository
        .observeEvents(configuration)
        .flatMap { event ->
            Logger.d().addArg(event).with(this@LocatorInteractorImpl).print()
            mExtensions["isCharging"] = event.battery.isCharging.toString()
            mExtensions["level"] = event.battery.level.toString()
            Observable.empty<Geo>()
        }

    private fun lbsEmitter(configuration: Configuration.LbsDataConfiguration): Observable<Geo> = mLbsRepository
        .observeEvents(configuration)
        .flatMapSingle { event ->
            Logger.d().addArg(event).with(this@LocatorInteractorImpl).print()
            mCurrentSessionRepository
                .getSession()
                .flatMap { session ->
                    mLastLbsList = event.lbsList
                    mYandexLocatorRepository
                        .get(mLastWifiList, mLastLbsList, Date())
                        .map<Geo> {
                            Geo.LbsLocatorLocationGeo(mLastLbsList, mLastWifiList, it, session, Date(), mExtensions)
                        }
                        .onErrorReturn {
                            Geo.LbsLocatorErrorGeo(mLastLbsList, mLastWifiList, it.toString(), session, Date(), mExtensions)
                        }
                }
        }

    private fun wifiEmitter(configuration: Configuration.WifiDataConfiguration): Observable<Geo> = mWifiRepository
        .observeEvents(configuration)
        .flatMapSingle { event ->
            Logger.d().addArg(event).with(this@LocatorInteractorImpl).print()
            mCurrentSessionRepository
                .getSession()
                .flatMap { session ->
                    mLastWifiList = event.wifiFullList
                    mYandexLocatorRepository
                        .get(mLastWifiList, mLastLbsList, event.create)
                        .map<Geo> {
                            Geo.WifiLocatorLocationGeo(mLastWifiList, mLastLbsList, it, session, Date(), mExtensions)
                        }
                        .onErrorReturn {
                            Geo.WifiLocatorErrorGeo(mLastWifiList, mLastLbsList, it.toString(), session, Date(), mExtensions)
                        }
                }
        }


    private fun isSendCriteria(geo: Geo, configuration: Configuration.LocatorLiveConfiguration): Boolean {
        Logger.d().addArg(geo).addArg(configuration).print()
        geo.getLocation()?.accuracy?.also {
            if (it > configuration.accuracyIgnoreCriteria) {
                Logger.d("Bad accuracy").print()
                return false
            }
        }
        if (mLastSendedGeo == null) {
            Logger.d("First geo").print()
            return true
        }
        if (mNonSended.size > configuration.countNotSendedGeoSendCriteria) {
            Logger.d("Many not sended").print()
            return true
        }
        val lastLocation = mLastSendedGeo!!.getLocation()
        val currentLocation = when (geo) {
            is Geo.LbsLocatorLocationGeo -> geo.getLocation()
            is Geo.LbsLocatorErrorGeo -> geo.getLocation()
            is Geo.WifiLocatorLocationGeo -> geo.getLocation()
            is Geo.WifiLocatorErrorGeo -> geo.getLocation()
        }
        if (currentLocation != null) {
            if (lastLocation == null) {
                Logger.d("First geo with location").print()
                return true
            }
            if (currentLocation.accuracy != null && lastLocation.accuracy != null &&
                currentLocation.accuracy / lastLocation.accuracy < configuration.accuracySendCriteria
            ) {
                // сменилась точность
                Logger.d("Accuracy changed").print()
                return true
            }
            if (Location.distanceBetween(lastLocation, currentLocation) > configuration.distanceSendCriteria) {
                // сместились достаточно от предыдущие отправленной кординаты
                Logger.d("Distance changed").print()
                return true
            }
            if (mNonSended.firstOrNull { notSended ->
                    notSended.getLocation()?.let {
                        Location.distanceBetween(it, currentLocation) > configuration.distanceSendCriteria
                    } == true
                } != null) {
                // сместились достаточно от не отправленной кординаты
                Logger.d("Distance changed from not sended").print()
                return true
            }
        } else {
            val currentError = geo.getError() ?: throw GeoException.InvalidLocatorGeo(geo)
            if (lastLocation != null) {
                // первая ошибка
                Logger.d("First error").print()
                return true
            }
            val lastError = mLastSendedGeo!!.getError()!!
            if (currentError != lastError) {
                // иная ошибка
                Logger.d("Another error").print()
                return true
            }
            if (mNonSended.firstOrNull { notSended -> notSended.getError()?.let { currentError != it } == true } != null) {
                // сменилась ошибка
                Logger.d("Error changed").print()
                return true
            }
        }
        return false
    }
}