package org.findmykids.geo.data.repository.trigger.timer.alarm

import android.app.AlarmManager
import android.app.AlarmManager.ELAPSED_REALTIME_WAKEUP
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.SystemClock
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.trigger.timer.TimerManager
import javax.inject.Inject


internal class TimerAlarmManagerImpl @Inject constructor(
    private val mContext: Context
) : TimerManager {

    override fun toString(): String = ""

    override fun isSubscribed(): Single<Boolean> = Single
        .fromCallable {
            Logger.d().with(this@TimerAlarmManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE) != null
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@TimerAlarmManagerImpl).print()
        }

    override fun subscribe(configuration: Configuration.TimerConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@TimerAlarmManagerImpl).print()
            getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                Logger.d("has intent").with(this@TimerAlarmManagerImpl).print()
                val timeToAlarm = SystemClock.elapsedRealtime() + configuration.delay
                val alarmManager = ContextCompat.getSystemService(mContext, AlarmManager::class.java)!!
                when {
                    Build.VERSION.SDK_INT >= 23 -> alarmManager.setExactAndAllowWhileIdle(ELAPSED_REALTIME_WAKEUP, timeToAlarm, intent)
                    Build.VERSION.SDK_INT >= 19 -> alarmManager.setExact(ELAPSED_REALTIME_WAKEUP, timeToAlarm, intent)
                    else -> alarmManager.set(ELAPSED_REALTIME_WAKEUP, timeToAlarm, intent)
                }
                true
            } ?: false
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@TimerAlarmManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@TimerAlarmManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
                Logger.d("has intent").with(this@TimerAlarmManagerImpl).print()
                ContextCompat.getSystemService(mContext, AlarmManager::class.java)!!.cancel(intent)
                intent.cancel()
                true
            } ?: true
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@TimerAlarmManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, AlarmManagerReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_TIMER, intent, flags)
    }


    companion object {
        private const val REQUEST_TIMER = 6748608
    }
}