package org.findmykids.geo.data.repository.live

import android.os.Build
import android.os.HandlerThread
import io.reactivex.Completable
import io.reactivex.Observable
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.BehaviorSubject
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.live.sensors.SensorsEvent


/**
 * Основной источних данных
 */
internal abstract class BaseLiveRepository<C : Configuration, T> {

    private var mDataBehaviorSubject: BehaviorSubject<T> = BehaviorSubject.create()
    private var mErrorBehaviorSubject: BehaviorSubject<GeoException> = BehaviorSubject.create()
    private var mHandlerThread: HandlerThread? = null


    override fun toString(): String = ""


    /**
     * Подписка на получение данных
     * Непосредственно запуск получения данных запустится после первой подписки,
     * последующие подписки повторно не запустят получения данных
     */
    fun observeEvents(configuration: C): Observable<T> = mDataBehaviorSubject
        .doOnSubscribe {
            Logger.d("doOnSubscribe").with(this@BaseLiveRepository).print()
            if (!mDataBehaviorSubject.hasObservers()) {
                Logger.d("start").with(this@BaseLiveRepository).print()
                mHandlerThread = HandlerThread("LiveDataThread").apply {
                    start()
                }
                start(configuration, mHandlerThread!!)
            }
        }
        .doFinally {
            Logger.d("doFinally").with(this@BaseLiveRepository).print()
            if (!mDataBehaviorSubject.hasObservers()) {
                Logger.d("stop").with(this@BaseLiveRepository).print()
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                    mHandlerThread?.quitSafely()
                } else {
                    mHandlerThread?.quit()
                }
                mHandlerThread = null
                stop()
            }
        }
        .doOnNext {
            if (it !is SensorsEvent) { // many logs
                Logger.d("doOnNext").addArg(it).with(this@BaseLiveRepository).print()
            }
        }
        .observeOn(Schedulers.io())

    /**
     * Подписка на получение ошибок
     */
    fun observeErrors(): Observable<GeoException> = mErrorBehaviorSubject
        .doOnNext {
            Logger.w(it).with(this@BaseLiveRepository).print()
        }
        .observeOn(Schedulers.io())


    /**
     * Перезапуск получения данных
     */
    fun restart(configuration: C): Completable = Completable
        .fromCallable {
            Logger.d().with(this@BaseLiveRepository).print()
            if (mDataBehaviorSubject.hasObservers()) {
                stop()
                start(configuration, mHandlerThread!!)
            }
        }


    protected fun sendEvent(t: T) {
        if (t !is SensorsEvent) { // many logs
            Logger.i().addArg(t).with(this).print()
        }
        mDataBehaviorSubject.onNext(t)
    }

    protected fun sendError(geoException: GeoException) {
        Logger.w(geoException).with(this).print()
        mErrorBehaviorSubject.onNext(geoException)
    }

    /**
     * Вернет запущено ли получение данных
     */
    protected fun isStarted() = mDataBehaviorSubject.hasObservers()


    /**
     * Запуск получения данных
     */
    protected abstract fun start(configuration: C, handlerThread: HandlerThread)

    /**
     * Остановка получения данных
     */
    protected abstract fun stop()
}