package org.findmykids.geo.domain.factory

import android.content.Intent
import com.google.gson.Gson
import org.findmykids.geo.BuildConfig
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.ActivityEvent
import org.findmykids.geo.data.model.Location
import org.findmykids.geo.data.model.StationEvent
import org.findmykids.geo.data.model.ZoneEvent
import org.findmykids.geo.data.repository.trigger.activity.BaseActivityReceiver
import org.findmykids.geo.data.repository.trigger.boot.BootReceiver
import org.findmykids.geo.data.repository.trigger.passive.BasePassiveReceiver
import org.findmykids.geo.data.repository.trigger.station.BaseStationReceiver
import org.findmykids.geo.data.repository.trigger.task.Tasks
import org.findmykids.geo.data.repository.trigger.timer.alarm.AlarmManagerReceiver
import org.findmykids.geo.data.repository.trigger.zones.BaseZonesReceiver
import org.findmykids.geo.domain.model.Command


/**
 * Фабрика создания команды из интента.
 * try нужен для того чтобы избежать кеширование интентов.
 * Ловил ситуацию когда я в коде уже менял параметры интента, а в момент запуска прилетал интент со старыми.
 */
internal object CommandFactory {
    fun createInnerCommand(intent: Intent) = try {
        when (intent.action) {
            AlarmManagerReceiver.ACTION_TIMER_ALARM -> {
                Logger.d("Timer Alarm").print()
                Command.Timer()
            }
            BasePassiveReceiver.ACTION_PASSIVE -> {
                Logger.d("Passive").print()
                val textLocation = intent.getStringExtra(BasePassiveReceiver.EXTRA_LOCATION)
                val location = Gson().fromJson(textLocation, Location::class.java)
                Command.Passive(location)
            }
            BaseActivityReceiver.ACTION_ACTIVITY -> {
                Logger.d("Activity").print()
                val textActivityTransitionsEvent = intent.getStringExtra(BaseActivityReceiver.EXTRA_ACTIVITY_EVENT)
                val activityTransitionsEvent = Gson().fromJson(textActivityTransitionsEvent, ActivityEvent::class.java)
                Command.Activity(activityTransitionsEvent)
            }
            BaseZonesReceiver.ACTION_ZONE -> {
                Logger.d("Zone").print()
                val textZoneEvent = intent.getStringExtra(BaseZonesReceiver.EXTRA_ZONE_EVENT)
                val zoneEvent = Gson().fromJson(textZoneEvent, ZoneEvent::class.java)
                Command.Zone(zoneEvent)
            }
            BaseStationReceiver.ACTION_STATION -> {
                Logger.d("Station").print()
                val textStationEvent = intent.getStringExtra(BaseStationReceiver.EXTRA_STATION_EVENT)
                val stationEvent = Gson().fromJson(textStationEvent, StationEvent::class.java)
                Command.Station(stationEvent)
            }
            BootReceiver.ACTION_BOOT -> {
                Logger.d("Boot").print()
                Command.Boot()
            }
            Tasks.ACTIVATE.name -> {
                Logger.d("Activate").print()
                Command.Activate(
                    intent.getStringExtra(Tasks.BASE_URL)!!,
                    intent.getCharArrayExtra(Tasks.API_KEY)!!,
                    intent.getCharArrayExtra(Tasks.USER_TOKEN)!!,
                    intent.getCharArrayExtra(Tasks.USER_ID)!!
                )
            }
            Tasks.DEACTIVATE.name -> {
                Logger.d("Deactivate").print()
                Command.Deactivate()
            }
            Tasks.PING.name -> {
                Logger.d("Ping").print()
                Command.Ping()
            }
            Tasks.REQUEST.name -> {
                Logger.d("Request").print()
                Command.Request()
            }
            Tasks.REAL_TIME.name -> {
                Logger.d("RealTime").print()
                Command.Realtime(intent.getBooleanExtra(Tasks.REAL_TIME_ENABLED, false))
            }
            else -> {
                Logger.d("Not supported action").print()
                throw GeoException.NotSupportedAction(intent.action)
            }
        }
    } catch (t: Throwable) {
        if (BuildConfig.DEBUG) {
            throw t
        } else {
            Command.Invalid(t)
        }
    }
}