package org.findmykids.geo.data.repository.live.remote

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.ConnectivityManager
import android.net.Network
import android.os.Build
import android.os.Handler
import android.os.HandlerThread
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.network.SocketClient
import org.findmykids.geo.data.preferences.LocalPreferences
import org.findmykids.geo.data.repository.storage.currentSession.CurrentSessionRepository
import java.util.concurrent.TimeUnit
import javax.inject.Inject
import kotlin.math.min


internal class RemoteRepositoryImpl @Inject constructor(
    private val mSocketClient: SocketClient,
    private val mContext: Context,
    private val mCurrentSessionRepository: CurrentSessionRepository,
    private val mLocalPreferences: LocalPreferences
) : RemoteRepository() {

    private lateinit var mConfiguration: Configuration.RemoteDataConfiguration
    private lateinit var mHandlerThread: HandlerThread

    private var mSocketDisposable: Disposable? = null
    private var mRestartDisposable: Disposable? = null
    private var mRetryCount = 0

    @Volatile
    private var mIsActiveListenNetworkState: Boolean = false

    private val mNetworkChangeReceiver: BroadcastReceiver by lazy {
        object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent) {
                Logger.d().print()
                reconnectIfNeed()
            }
        }
    }

    private val mNetworkCallback: ConnectivityManager.NetworkCallback by lazy {
        @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
        object : ConnectivityManager.NetworkCallback() {
            override fun onAvailable(network: Network) {
                Logger.d().print()
                reconnectIfNeed()
            }

            override fun onLost(network: Network) {}
        }
    }


    override fun toString(): String = ""

    override fun start(configuration: Configuration.RemoteDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        mConfiguration = configuration
        mHandlerThread = handlerThread
        mRestartDisposable?.dispose()
        mRestartDisposable = null
        mSocketDisposable = Observable
            .create(mSocketClient)
            .subscribe({ socketData ->
                Logger.d().addArg(socketData).with(this@RemoteRepositoryImpl).print()
                mRetryCount = 0
                try {
                    sendEvent(RemoteEventFactory.createSocketEvent(socketData))
                } catch (e: Exception) {
                    sendError(GeoException.InvalidSocketData(e))
                }
            }, {
                if (it is GeoException) {
                    sendError(it)
                    if (it is GeoException.SocketInvalidUserId) {
                        mLocalPreferences.setUserId(it.userId)
                    }
                } else {
                    sendError(GeoException.Socket(it))
                }
                restart(configuration, handlerThread)
            })
        if (!mIsActiveListenNetworkState) {
            startListenNetworkState(handlerThread)
        }
    }

    override fun stop() {
        Logger.i().print()
        if (mIsActiveListenNetworkState) {
            stopListenNetworkState()
        }
        mSocketDisposable?.dispose()
        mSocketDisposable = null
        mRestartDisposable?.dispose()
        mRestartDisposable = null
    }


    @Synchronized
    private fun reconnectIfNeed() {
        Logger.d().print()
        if (mSocketDisposable == null) {
            mRestartDisposable?.dispose()
            mRestartDisposable = null
            if (isStarted()) {
                start(mConfiguration, mHandlerThread)
            }
        }
    }

    @Synchronized
    private fun restart(configuration: Configuration.RemoteDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mSocketDisposable = null
        mRestartDisposable = mCurrentSessionRepository
            .getSession()
            .map {
                Logger.d().addArg(it).with(this@RemoteRepositoryImpl).print()
                val delay = if (it.isRealtime) {
                    configuration.realtimeDelay
                } else {
                    configuration.startDelay + mRetryCount * configuration.stepDelay
                }
                mRetryCount++
                min(delay, configuration.maxDelay)
            }
            .flatMapCompletable {
                Logger.d().addArg(it).with(this@RemoteRepositoryImpl).print()
                Completable.timer(it, TimeUnit.MILLISECONDS, Schedulers.newThread())
            }
            .subscribe({
                mRestartDisposable = null
                Logger.d("restart").with(this@RemoteRepositoryImpl).print()
                if (isStarted()) {
                    start(configuration, handlerThread)
                }
            }, {
                mRestartDisposable = null
                sendError(GeoException.SocketRestart(it))
                restart(configuration, handlerThread)
            })
    }


    private fun startListenNetworkState(handlerThread: HandlerThread) {
        Logger.d().print()
        mIsActiveListenNetworkState = true
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val connectivityManager = ContextCompat.getSystemService(mContext, ConnectivityManager::class.java)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                connectivityManager?.registerDefaultNetworkCallback(mNetworkCallback, Handler(handlerThread.looper))
            } else {
                connectivityManager?.registerDefaultNetworkCallback(mNetworkCallback)
            }
        } else {
            val intentFilter = IntentFilter()
            @Suppress("DEPRECATION")
            intentFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION)
            mContext.registerReceiver(mNetworkChangeReceiver, intentFilter, null, Handler(handlerThread.looper))
        }
    }

    private fun stopListenNetworkState() {
        Logger.d().print()
        mIsActiveListenNetworkState = false
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val connectivityManager = ContextCompat.getSystemService(mContext, ConnectivityManager::class.java)
            connectivityManager?.unregisterNetworkCallback(mNetworkCallback)
        } else {
            mContext.unregisterReceiver(mNetworkChangeReceiver)
        }
    }
}