package org.findmykids.geo.api

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.database.Cursor
import android.net.Uri
import android.os.Build
import org.findmykids.geo.api.model.GeoPlatformState
import org.findmykids.geo.data.repository.trigger.task.Tasks
import org.findmykids.geo.presentation.manage.AuthorityCheckProvider
import org.findmykids.geo.presentation.session.SessionService
import java.lang.IllegalStateException


/**
 * Основной класс для управления гео платформы
 * Общение происходит через startService, startForegroundService и WorkManager в зависимости от условий
 */
object GeoPlatform {

    /**
     * Узнать состояние гео платформы
     */
    @SuppressLint("Recycle")
    @JvmStatic
    @Synchronized
    fun isActivatedTo(context: Context, baseUrl: String, userId: CharArray): GeoPlatformState {
        var cursor: Cursor? = null

        return try {
            cursor = context.contentResolver.query(
                Uri.parse("content://" + GeoPlatformSettings.mAppId + "." + AuthorityCheckProvider.NAME),
                null,
                null,
                arrayOf(baseUrl, String(userId)),
                null
            )

            if (cursor == null) {
                throw IllegalStateException("Cursor should not be null")
            }

            if (!cursor.moveToFirst()) {
                throw IllegalStateException("Cursor should not be empty")
            }

            if(cursor.columnCount <=0 ){
                throw IllegalStateException("Cursor should have more than 0 columns")
            }

            ping(context)

            GeoPlatformState.getState(cursor.getInt(0))

        } catch (e: Exception) {
            e.printStackTrace()
            GeoPlatformState.ERROR(e)
        } finally {
            try {
                cursor?.close()
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    /**
     * Активация гео платформы
     */
    @JvmStatic
    fun activate(context: Context, baseUrl: String, apiKey: CharArray, userToken: CharArray, userId: CharArray) {
        startService(
            context,
            getIntent(context, Tasks.ACTIVATE).apply {
                putExtra(Tasks.API_KEY, apiKey)
                putExtra(Tasks.USER_ID, userId)
                putExtra(Tasks.BASE_URL, baseUrl)
                putExtra(Tasks.USER_TOKEN, userToken)
            }
        )
    }

    /**
     * деактивация геоплатформы
     */
    @JvmStatic
    fun deactivate(context: Context) {
        startService(context, getIntent(context, Tasks.DEACTIVATE))
    }


    /**
     * Запустить геопалторму
     */
    @JvmStatic
    fun ping(context: Context) {
        startService(context, getIntent(context, Tasks.PING))
    }


    /**
     * Запрос координаты и посылает ее на сервер. не используется, поэтому может не работать
     */
    @JvmStatic
    fun locationRequest(context: Context) {
        startService(context, getIntent(context, Tasks.REQUEST))
    }

    /**
     * Включить или отключить реалтайм
     */
    @JvmStatic
    fun setRealtime(context: Context, enable: Boolean) {
        startService(
            context,
            getIntent(context, Tasks.REAL_TIME).apply {
                putExtra(Tasks.REAL_TIME_ENABLED, enable)
            }
        )
    }


    private fun getIntent(context: Context, task: Tasks) =
        Intent(context, SessionService::class.java).apply {
            action = task.name
        }

    private fun startService(context: Context, intent: Intent) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            context.startForegroundService(intent)
        } else {
            context.startService(intent)
        }
    }
}