package org.findmykids.geo.data.repository.trigger.station.google

import android.annotation.SuppressLint
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import com.google.android.gms.location.Geofence
import com.google.android.gms.location.GeofencingRequest
import com.google.android.gms.location.LocationServices
import com.google.android.gms.tasks.Tasks
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Location
import org.findmykids.geo.data.repository.trigger.station.StationManager
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class StationGoogleManagerImpl @Inject constructor(
    private val mContext: Context
) : StationManager {

    override fun toString(): String = ""


    @SuppressLint("MissingPermission")
    override fun subscribe(location: Location, configuration: Configuration.StationConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@StationGoogleManagerImpl).print()
            getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                Logger.d("has intent").with(this@StationGoogleManagerImpl).print()
                val geofencingRequest = GeofencingRequest.Builder()
                    .setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_EXIT)
                    .addGeofence(
                        Geofence.Builder()
                            .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_EXIT)
                            .setCircularRegion(location.latitude, location.longitude, configuration.lowRadius)
                            .setRequestId(GEOFENCE_STATION_ID_LOW)
                            .setExpirationDuration(Geofence.NEVER_EXPIRE)
                            .build()
                    )
                    .addGeofence(
                        Geofence.Builder()
                            .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_EXIT)
                            .setCircularRegion(location.latitude, location.longitude, configuration.highRadius)
                            .setRequestId(GEOFENCE_STATION_ID_HIGH)
                            .setExpirationDuration(Geofence.NEVER_EXPIRE)
                            .build()
                    )
                    .build()
                try {
                    Tasks.await(LocationServices.getGeofencingClient(mContext).addGeofences(geofencingRequest, intent), 10, TimeUnit.SECONDS)
                    true
                } catch (e: Exception) {
                    Logger.e(GeoException.StationSubscription(e)).with(this@StationGoogleManagerImpl).print()
                    false
                }
            } ?: false
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@StationGoogleManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@StationGoogleManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
                Logger.d("has intent").with(this@StationGoogleManagerImpl).print()
                try {
                    Tasks.await(
                        LocationServices.getGeofencingClient(mContext).removeGeofences(
                            listOf(
                                GEOFENCE_STATION_ID_LOW,
                                GEOFENCE_STATION_ID_HIGH
                            )
                        ), 10, TimeUnit.SECONDS
                    )
                    Tasks.await(LocationServices.getGeofencingClient(mContext).removeGeofences(intent), 10, TimeUnit.SECONDS)
                    true
                } catch (e: Exception) {
                    Logger.e(GeoException.StationUnSubscription(e)).with(this@StationGoogleManagerImpl).print()
                    false
                } finally {
                    intent.cancel()
                }
            } ?: true
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@StationGoogleManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, StationGoogleReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_STATION_GOOGLE, intent, flags)
    }


    companion object {
        private const val REQUEST_STATION_GOOGLE = 6748606
        private const val GEOFENCE_STATION_ID_LOW = "GeoStation#blabla#34563026"
        private const val GEOFENCE_STATION_ID_HIGH = "GeoStation#blabla#75743135"
    }
}