package org.findmykids.geo.data.repository.trigger.passive.locationmanager

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.location.LocationManager
import android.os.Build
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.trigger.passive.PassiveManager
import javax.inject.Inject


internal class PassiveLocationManagerManagerImpl @Inject constructor(
    private val mContext: Context
) : PassiveManager {

    override fun toString(): String = ""

    override fun isSubscribed(): Single<Boolean> = Single
        .fromCallable {
            Logger.d().with(this@PassiveLocationManagerManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE) != null
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveLocationManagerManagerImpl).print()
        }

    override fun subscribe(configuration: Configuration.PassiveConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@PassiveLocationManagerManagerImpl).print()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&
                mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
                mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
            ) {
                Logger.w(GeoException.PassiveNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION))
                    .with(this@PassiveLocationManagerManagerImpl).print()
                false
            } else {
                getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { pendingIntent ->
                    Logger.d("has intent").with(this@PassiveLocationManagerManagerImpl).print()
                    ContextCompat.getSystemService(mContext, LocationManager::class.java)!!.requestLocationUpdates(
                        LocationManager.PASSIVE_PROVIDER,
                        configuration.minTime,
                        configuration.minDistance,
                        pendingIntent
                    )
                    true
                } ?: false
            }
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveLocationManagerManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@PassiveLocationManagerManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE)?.let { pendingIntent ->
                Logger.d("has intent").with(this@PassiveLocationManagerManagerImpl).print()
                ContextCompat.getSystemService(mContext, LocationManager::class.java)!!.removeUpdates(pendingIntent)
                pendingIntent.cancel()
                true
            } ?: true
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveLocationManagerManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, PassiveLocationManagerReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_PASSIVE_LOCATION_MANAGER, intent, flags)
    }


    companion object {
        private const val REQUEST_PASSIVE_LOCATION_MANAGER = 6748605
    }
}