package org.findmykids.geo.common.logger

import android.text.TextUtils
import android.util.Log
import io.reactivex.annotations.CheckReturnValue
import org.findmykids.geo.BuildConfig
import java.text.SimpleDateFormat
import java.util.*


internal class LogBuilder(private val mType: LogType, private val mMessage: String? = null) {
    private val mArgs: MutableList<Any?> = mutableListOf()
    private var mThis: Any? = null
    private var mResult: Any? = null
    private var mHasResult: Boolean = false


    override fun toString(): String = ""


    @CheckReturnValue
    fun with(ths: Any): LogBuilder {
        mThis = ths
        return this
    }


    @CheckReturnValue
    fun addArg(value: Any?): LogBuilder {
        mArgs.add(value)
        return this
    }


    @CheckReturnValue
    fun setResult(result: Any?): LogBuilder {
        mResult = result
        mHasResult = true
        return this
    }


    fun print() {
        val date = mDateFormat.format(Date())

        val pid = Thread.currentThread().id

        val fullClassName: String
        val methodName: String
        val element = Thread.currentThread().stackTrace[3]
        if (mThis == null) {
            fullClassName = element.className
            methodName = element.methodName
        } else {
            fullClassName = mThis!!.javaClass.canonicalName!!
            methodName = if (element.className.contains("$")) {
                element.className.removeRange(0, element.className.indexOf("$") + 1)
            } else {
                element.methodName
            }
        }
        @Suppress("ConstantConditionIf")
        val className = if (BuildConfig.MINIFY_ENABLED) {
            fullClassName
        } else {
            fullClassName.substring(BuildConfig.LIBRARY_PACKAGE_NAME.length + 1)
        }

        val args = TextUtils.join(", ", mArgs.map { "${it ?: "null"}" })

        val result = if (mHasResult) "= ${if (mResult == null) "null" else mResult} " else ""

        val message = if (mMessage == null) "" else "- $mMessage"

        val suffix = "($args) $result$message"

        val index = mIndex++
        val tag = "$TAG($index): $pid $date"
        val text = "$className.$methodName$suffix"
        if (enableLogs) {
            when (mType) {
                LogType.DEBUG -> Log.d(tag, text)
                LogType.INFO -> Log.i(tag, text)
                LogType.WARNING -> Log.w(tag, text)
                LogType.ERROR -> Log.e(tag, text)
            }
        }

        DBLogSaver.save(pid, date, mType, className, methodName, suffix)
        FileLogSaver.save(tag, text)
    }


    companion object {
        private val mDateFormat = SimpleDateFormat("yyyy-MM-dd: HH:mm:ss.SSSS", Locale.ENGLISH)
        private const val TAG = "Geo"
        private var mIndex: Long = 1

        var enableLogs = false
    }
}