package org.findmykids.geo.presentation.session.input

import io.reactivex.Observable
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.domain.authorization.AuthorizationInteractor
import org.findmykids.geo.domain.model.Command
import org.findmykids.geo.domain.session.SessionInteractor
import org.findmykids.geo.domain.subscriber.activity.ActivitySubscriberInteractor
import org.findmykids.geo.domain.subscriber.passive.PassiveSubscriberInteractor
import org.findmykids.geo.domain.subscriber.station.StationSubscriberInteractor
import org.findmykids.geo.domain.subscriber.timer.TimerSubscriberInteractor
import org.findmykids.geo.domain.subscriber.zones.ZonesSubscriberInteractor
import javax.inject.Inject


internal class CommandFrontControllerImpl @Inject constructor(
    private val mAuthorizationInteractor: AuthorizationInteractor,
    private val mSessionInteractor: SessionInteractor,
    private val mTimerSubscriberInteractor: TimerSubscriberInteractor,
    private val mPassiveSubscriberInteractor: PassiveSubscriberInteractor,
    private val mZonesSubscriberInteractor: ZonesSubscriberInteractor,
    private val mActivitySubscriberInteractor: ActivitySubscriberInteractor,
    private val mStationSubscriberInteractor: StationSubscriberInteractor
) : CommandFrontController {

    override fun toString(): String = ""

    /**
     * При добавлении новой команды заного подписываемся на ряд тригеров, на какие зависит от команды
     */
    override fun add(command: Command): Observable<Pair<Boolean, Int>> {
        Logger.i().addArg(command).print()
        return mSessionInteractor
            .addCommand(command)
            .andThen(
                when (command) {
                    is Command.Activate -> activate(command.baseUrl, command.apiKey, command.userToken, command.userId)
                    is Command.Deactivate -> deactivate()
                    is Command.Ping -> ping()
                    is Command.Boot -> boot()
                    is Command.Request -> request()
                    is Command.Timer -> timer()
                    is Command.Activity -> activity()
                    is Command.Zone -> zone()
                    is Command.Station -> station()
                    is Command.Passive -> passive()
                    is Command.Realtime -> setRealtime(command.enable)
                    is Command.Invalid -> ping()
                }
            )
            .doOnNext {
                Logger.d().with(this@CommandFrontControllerImpl).setResult(it).print()
            }
    }


    private fun activate(
        baseUrl: String,
        apiKey: CharArray,
        userToken: CharArray,
        userId: CharArray
    ): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .activate(baseUrl, apiKey, userToken, userId)
        .andThen(mTimerSubscriberInteractor.subscribe())
        .andThen(mPassiveSubscriberInteractor.subscribe())
        .andThen(mZonesSubscriberInteractor.subscribe())
        .andThen(mActivitySubscriberInteractor.subscribe())
        .andThen(Observable.just(Pair(false, 1), Pair(true, 0)))

    private fun deactivate(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .deactivate()
        .andThen(mTimerSubscriberInteractor.unsubscribe())
        .andThen(mPassiveSubscriberInteractor.unsubscribe())
        .andThen(mZonesSubscriberInteractor.unsubscribe())
        .andThen(mActivitySubscriberInteractor.unsubscribe())
        .andThen(mStationSubscriberInteractor.unsubscribe())
        .andThen(Observable.just(Pair(false, 0)))

    private fun ping(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribe()
                    .andThen(mPassiveSubscriberInteractor.subscribe())
                    .andThen(mZonesSubscriberInteractor.subscribe())
                    .andThen(mActivitySubscriberInteractor.subscribe())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun boot(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribe()
                    .andThen(mPassiveSubscriberInteractor.subscribe())
                    .andThen(mZonesSubscriberInteractor.subscribe())
                    .andThen(mActivitySubscriberInteractor.subscribe())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun timer(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribe()
                    .andThen(mPassiveSubscriberInteractor.subscribe())
                    .andThen(mZonesSubscriberInteractor.subscribe())
                    .andThen(mActivitySubscriberInteractor.subscribe())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    // TODO Зоны безопасности
    private fun zone(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribeIfNeed()
                    .andThen(mPassiveSubscriberInteractor.subscribeIfNeed())
                    .andThen(mZonesSubscriberInteractor.subscribe())
                    .andThen(mActivitySubscriberInteractor.subscribeIfNeed())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun station(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribeIfNeed()
                    .andThen(mPassiveSubscriberInteractor.subscribeIfNeed())
                    .andThen(mZonesSubscriberInteractor.subscribeIfNeed())
                    .andThen(mActivitySubscriberInteractor.subscribeIfNeed())
                    .andThen(mStationSubscriberInteractor.unsubscribe())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun passive(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribeIfNeed()
                    .andThen(mPassiveSubscriberInteractor.subscribeIfNeed())
                    .andThen(mZonesSubscriberInteractor.subscribeIfNeed())
                    .andThen(mActivitySubscriberInteractor.subscribeIfNeed())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun activity(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribeIfNeed()
                    .andThen(mPassiveSubscriberInteractor.subscribeIfNeed())
                    .andThen(mZonesSubscriberInteractor.subscribeIfNeed())
                    .andThen(mActivitySubscriberInteractor.subscribeIfNeed())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun request(): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribe()
                    .andThen(mPassiveSubscriberInteractor.subscribe())
                    .andThen(mZonesSubscriberInteractor.subscribe())
                    .andThen(mActivitySubscriberInteractor.subscribe())
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }

    private fun setRealtime(isEnable: Boolean): Observable<Pair<Boolean, Int>> = mAuthorizationInteractor
        .isActivated()
        .flatMapObservable { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribeIfNeed()
                    .andThen(mPassiveSubscriberInteractor.subscribeIfNeed())
                    .andThen(mZonesSubscriberInteractor.subscribeIfNeed())
                    .andThen(mActivitySubscriberInteractor.subscribeIfNeed())
                    .andThen(mSessionInteractor.setRealtime(isEnable))
                    .andThen(Observable.just(Pair(true, 0)))
            } else {
                Observable.just(Pair(false, 0))
            }
        }
}