package org.findmykids.geo.domain.live.timeout

import io.reactivex.Observable
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Activity
import org.findmykids.geo.data.model.Battery
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Error
import org.findmykids.geo.data.repository.live.activity.ActivityRepository
import org.findmykids.geo.data.repository.live.battery.BatteryRepository
import org.findmykids.geo.data.repository.live.location.LocationRepository
import org.findmykids.geo.data.repository.live.timeout.TimeoutRepository
import org.findmykids.geo.data.repository.storage.configuration.ConfigurationRepository
import org.findmykids.geo.data.repository.storage.currentSession.CurrentSessionRepository
import org.findmykids.geo.data.repository.storage.error.ErrorRepository
import org.findmykids.geo.domain.BaseInteractor
import org.findmykids.geo.domain.model.InnerEvent
import java.util.*
import javax.inject.Inject


internal class TimeoutInteractorImpl @Inject constructor(
    private val mConfigurationRepository: ConfigurationRepository,
    private val mCurrentSessionRepository: CurrentSessionRepository,
    private val mErrorRepository: ErrorRepository,
    private val mTimeoutRepository: TimeoutRepository,
    private val mActivityRepository: ActivityRepository,
    private val mBatteryRepository: BatteryRepository,
    private val mLocationRepository: LocationRepository
) : BaseInteractor(), TimeoutInteractor {

    private var mBattery: Battery? = null
    private var mActivity: Activity? = null

    /**
     * Количество локаций после последнее сообщения о срабатывании таймера
     */
    @Volatile
    private var mBetweenTickLocationCounter = 0

    /**
     *  Общее количество полученных локаций
     */
    @Volatile
    private var mCommonLocationCounter = 0


    override fun toString(): String = ""

    override fun update(): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .doOnSubscribe {
            Logger.d().with(this@TimeoutInteractorImpl).print()
            mActivity = null
            mBattery = null
        }
        .flatMapObservable {
            // подписка
            Logger.d().with(this@TimeoutInteractorImpl).print()
            Observable.merge(
                activityEmitter(it.activityDataConfiguration),
                batteryEmitter(it.batteryDataConfiguration),
                locationEmitter(it.locationDataConfiguration),
                timeoutEmitter(it.timeoutDataConfiguration)
            )
        }
        .filter {
            // только событие от таймера обрабатываются далее
            it
        }
        .flatMap {
            when {
                mCommonLocationCounter == 0 -> mCurrentSessionRepository
                    .getSession()
                    .flatMapCompletable {
                        // отправляем сообщение о том что не получили координат
                        mErrorRepository.send(Error(Date(), it, mActivity, mBattery, GeoException.EmptyLocation().toString()))
                    }
                    .andThen(Observable.just(InnerEvent.EndCurrentSession(InnerEvent.EndCurrentSession.EndSessionReason.Timeout)))
                mBetweenTickLocationCounter == 0 -> Observable
                    .just(InnerEvent.EndCurrentSession(InnerEvent.EndCurrentSession.EndSessionReason.Timeout))
                else -> {
                    mBetweenTickLocationCounter = 0
                    Observable.empty<InnerEvent>()
                }
            }
        }


    private fun activityEmitter(configuration: Configuration.ActivityDataConfiguration): Observable<Boolean> = mActivityRepository
        .observeEvents(configuration)
        .map { event ->
            Logger.d().addArg(event).with(this@TimeoutInteractorImpl).print()
            mActivity = event.activity
            false
        }

    private fun batteryEmitter(configuration: Configuration.BatteryDataConfiguration): Observable<Boolean> = mBatteryRepository
        .observeEvents(configuration)
        .map { event ->
            Logger.d().addArg(event).with(this@TimeoutInteractorImpl).print()
            mBattery = event.battery
            false
        }

    private fun locationEmitter(configuration: Configuration.LocationDataConfiguration): Observable<Boolean> = mLocationRepository
        .observeEvents(configuration)
        .map { event ->
            Logger.d().addArg(event).with(this@TimeoutInteractorImpl).print()
            mCommonLocationCounter++
            mBetweenTickLocationCounter++
            false
        }

    private fun timeoutEmitter(configuration: Configuration.TimeoutDataConfiguration): Observable<Boolean> = mTimeoutRepository
        .observeEvents(configuration)
        .map { event ->
            Logger.d().addArg(event).with(this@TimeoutInteractorImpl).print()
            true
        }
}