package org.findmykids.geo.data.repository.trigger.passive.googlefused

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationServices
import com.google.android.gms.tasks.Tasks
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.trigger.passive.PassiveManager
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class PassiveGoogleFusedManagerImpl @Inject constructor(
    private val mContext: Context
) : PassiveManager {

    override fun toString(): String = ""

    override fun isSubscribed(): Single<Boolean> = Single
        .fromCallable {
            Logger.d().with(this@PassiveGoogleFusedManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE) != null
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveGoogleFusedManagerImpl).print()
        }

    override fun subscribe(configuration: Configuration.PassiveConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@PassiveGoogleFusedManagerImpl).print()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&
                mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
                mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
            ) {
                Logger.e(GeoException.PassiveNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION))
                    .with(this@PassiveGoogleFusedManagerImpl)
                    .print()
                false
            } else {
                getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                    Logger.d("has intent").with(this@PassiveGoogleFusedManagerImpl).print()
                    val locationRequest = LocationRequest()
                        .setInterval(configuration.minTime)
                        .setSmallestDisplacement(configuration.minDistance)
                        .setPriority(configuration.priority)
                    try {
                        Tasks.await(
                            LocationServices.getFusedLocationProviderClient(mContext).requestLocationUpdates(locationRequest, intent),
                            10,
                            TimeUnit.SECONDS
                        )
                        true
                    } catch (e: Exception) {
                        Logger.e(GeoException.PassiveSubscription(e)).with(this@PassiveGoogleFusedManagerImpl).print()
                        false
                    }
                } ?: false
            }
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveGoogleFusedManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@PassiveGoogleFusedManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
                Logger.d("has intent").with(this@PassiveGoogleFusedManagerImpl).print()
                try {
                    Tasks.await(LocationServices.getFusedLocationProviderClient(mContext).removeLocationUpdates(intent), 10, TimeUnit.SECONDS)
                    true
                } catch (e: Exception) {
                    Logger.e(GeoException.PassiveUnSubscription(e)).with(this@PassiveGoogleFusedManagerImpl).print()
                    false
                } finally {
                    intent.cancel()
                }
            } ?: true
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveGoogleFusedManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, PassiveGoogleFusedReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_PASSIVE_GOOGLE_FUSED, intent, flags)
    }


    companion object {
        private const val REQUEST_PASSIVE_GOOGLE_FUSED = 6748603
    }
}