package org.findmykids.geo.data.repository.trigger.activity.google

import android.Manifest.permission.ACTIVITY_RECOGNITION
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager.PERMISSION_GRANTED
import android.os.Build
import com.google.android.gms.location.ActivityRecognition
import com.google.android.gms.location.ActivityTransition
import com.google.android.gms.location.ActivityTransitionRequest
import com.google.android.gms.tasks.Tasks
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.trigger.activity.ActivityManager
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class ActivityGoogleManagerImpl @Inject constructor(
    private val mContext: Context
) : ActivityManager {

    override fun toString(): String = ""

    override fun isSubscribed(): Single<Boolean> = Single
        .fromCallable {
            Logger.d().with(this@ActivityGoogleManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE) != null
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@ActivityGoogleManagerImpl).print()
        }

    override fun subscribe(activityConfiguration: Configuration.ActivityConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@ActivityGoogleManagerImpl).print()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q &&
                mContext.checkSelfPermission(ACTIVITY_RECOGNITION) != PERMISSION_GRANTED
            ) {
                Logger.e(GeoException.ActivityTransitionNoPermission(ACTIVITY_RECOGNITION)).with(this@ActivityGoogleManagerImpl).print()
                false
            } else {
                getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                    Logger.d("has intent").with(this@ActivityGoogleManagerImpl).print()
                    val transitions = mutableListOf<ActivityTransition>()
                    for (activity in activityConfiguration.activities) {
                        transitions.add(ActivityTransition.Builder().setActivityType(activity).setActivityTransition(ActivityTransition.ACTIVITY_TRANSITION_ENTER).build())
                        transitions.add(ActivityTransition.Builder().setActivityType(activity).setActivityTransition(ActivityTransition.ACTIVITY_TRANSITION_EXIT).build())
                    }
                    try {
                        Tasks.await(
                            ActivityRecognition.getClient(mContext).requestActivityTransitionUpdates(
                                ActivityTransitionRequest(transitions),
                                intent
                            ), 10, TimeUnit.SECONDS
                        )
                        true
                    } catch (e: Exception) {
                        Logger.e(GeoException.ActivityTransitionSubscription(e)).with(this@ActivityGoogleManagerImpl).print()
                        false
                    }
                } ?: false
            }
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@ActivityGoogleManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@ActivityGoogleManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
                Logger.d("has intent").with(this@ActivityGoogleManagerImpl).print()
                try {
                    Tasks.await(ActivityRecognition.getClient(mContext).removeActivityTransitionUpdates(intent), 10, TimeUnit.SECONDS)
                    true
                } catch (e: Exception) {
                    Logger.e(GeoException.ActivityTransitionUnSubscription(e)).with(this@ActivityGoogleManagerImpl).print()
                    false
                } finally {
                    intent.cancel()
                }
            } ?: true
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@ActivityGoogleManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, ActivityGoogleReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_ACTIVITY_GOOGLE, intent, flags)
    }


    companion object {
        private const val REQUEST_ACTIVITY_GOOGLE = 6748601
    }
}