package org.findmykids.geo.data.repository.storage.geo

import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo._todo.Geo
import org.findmykids.geo.api.extensions.GeoExtensions
import org.findmykids.geo.common.Container
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.DateTimeUtil.differentInMinutes
import org.findmykids.geo.data.db.GeoDatabase
import org.findmykids.geo.data.db.factory.SendGeoLocationEntityFactory
import org.findmykids.geo.data.db.model.GeoEntity
import org.findmykids.geo.data.db.model.SendGeoLocationEntity
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.GeoLocation
import org.findmykids.geo.data.model.Session
import org.findmykids.geo.data.network.SocketClient
import org.findmykids.geo.data.network.factory.CoordsFactory
import org.findmykids.geo.data.network.model.SocketData
import org.findmykids.geo.data.preferences.LocalPreferences
import java.util.*
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class GeoRepositoryImpl @Inject constructor(
    private val mSocketClient: SocketClient,
    private val mGeoDatabase: GeoDatabase,
    private val mGeoExtensions: GeoExtensions,
    private val mLocalPreferences: LocalPreferences
) : GeoRepository {

    @Volatile
    private var mLastSendGeoLocation: GeoLocation? = null

    @Volatile
    private var mLastSaveGeoLocation: GeoLocation? = null

    @Volatile
    private var mIndex = 1L


    override fun toString(): String = ""

    override fun getLastSaveGeoLocation(): Single<Container<GeoLocation>> = Single
        .fromCallable {
            Logger.d().setResult(mLastSaveGeoLocation).with(this@GeoRepositoryImpl).print()
            Container(mLastSaveGeoLocation)
        }
        .doOnSuccess {
            Logger.d().setResult(it.value).with(this@GeoRepositoryImpl).print()
        }

    override fun saveGeoLocation(geoLocation: GeoLocation): Completable = Completable
        .fromCallable {
            Logger.d().addArg(geoLocation).with(this@GeoRepositoryImpl).print()
            mLastSaveGeoLocation = geoLocation
            Unit
        }
        .doOnComplete {
            Logger.d("Complete").with(this@GeoRepositoryImpl).print()
        }


    override fun getLastSendGeoLocation(): Single<Container<GeoLocation>> = Single
        .fromCallable {
            Logger.d().setResult(mLastSendGeoLocation).print()
            Container(mLastSendGeoLocation)
        }
        .doOnSuccess {
            Logger.d().setResult(it.value).with(this@GeoRepositoryImpl).print()
        }

    override fun sendGeoLocation(geoLocation: GeoLocation, reason: String): Completable = Single
        .fromCallable {
            Logger.d().addArg(reason).addArg(geoLocation).with(this@GeoRepositoryImpl).print()
            mLastSendGeoLocation = geoLocation
            SendGeoLocationFactory.create(geoLocation, reason, mIndex++, mLocalPreferences.getAndIncreaseSendGlobalIndex())
        }
        .flatMapCompletable { sendGeoLocation ->
            Logger.d().addArg(sendGeoLocation).with(this@GeoRepositoryImpl).print()
            mSocketClient
                .send(SocketData(CoordsFactory.createCoords(mGeoExtensions, true, sendGeoLocation).toByteArray()))
                .doOnSuccess { isSended ->
                    Logger.d().addArg(isSended).with(this@GeoRepositoryImpl).print()
                    if (!isSended) {
                        synchronized(mGeoDatabase.lock) {
                            mGeoDatabase
                                .sendGeoLocationsDao()
                                .insert(SendGeoLocationEntityFactory.create(sendGeoLocation))
                        }
                    }
                }
                .ignoreElement()
        }
        .doOnComplete {
            Logger.d("Complete").with(this@GeoRepositoryImpl).print()
        }


    override fun sendAll(session: Session, configuration: Configuration.GeoStorageConfiguration): Completable = Completable
        .fromCallable {
            Logger.d().addArg(configuration).with(this@GeoRepositoryImpl).print()
            var second = false
            do {
                if (second) {
                    try {
                        Thread.sleep(configuration.timeoutOnSendOffline)
                    } catch (e: Exception) {
                    }
                }
                val entities: List<GeoEntity> = synchronized(mGeoDatabase.lock) {
                    try {
                        mGeoDatabase
                            .geosDao()
                            .select(configuration.offlineMaxCount)
                    } catch (e: Exception) {
                        Logger.e(e).with(this@GeoRepositoryImpl).print()
                        mGeoDatabase
                            .geosDao()
                            .deleteAll()
                        listOf()
                    }
                }
                val onlineGeos = mutableListOf<GeoEntity>()
                val offlineGeos = mutableListOf<GeoEntity>()
                entities.forEach {
                    if (it.geo.session.create.time == session.create.time && differentInMinutes(Date(), it.geo.create) < 1) {
                        onlineGeos.add(it)
                    } else {
                        offlineGeos.add(it)
                    }
                }
                if (onlineGeos.isNotEmpty()) {
                    realSend(true, onlineGeos)
                }
                if (offlineGeos.isNotEmpty()) {
                    realSend(false, offlineGeos)
                }
                second = true
            } while (entities.isNotEmpty())
        }
        .toSingle {
            Logger.d("online").with(this@GeoRepositoryImpl).print()
            synchronized(mGeoDatabase.lock) {
                mGeoDatabase.sendGeoLocationsDao().selectOnline(session.index, System.currentTimeMillis())
            }
        }
        .flatMap { entities ->
            Logger.d().addArg(entities.size).with(this@GeoRepositoryImpl).print()
            if (entities.isNotEmpty()) {
                val locations = entities.map { SendGeoLocationFactory.create(it) }
                val coords = CoordsFactory.createCoords(mGeoExtensions, true, * locations.toTypedArray())
                mSocketClient
                    .send(SocketData(coords.toByteArray()))
                    .doOnSuccess { isSended ->
                        Logger.d().addArg(isSended).with(this@GeoRepositoryImpl).print()
                        if (isSended) {
                            removeSendGeoLocations(entities)
                        }
                    }
            } else {
                Single.just(false)
            }
        }
        .flatMapCompletable {
            val count = mGeoDatabase.sendGeoLocationsDao().getRowCount()
            Logger.d("offline").addArg(count).with(this@GeoRepositoryImpl).print()
            var isLastSended = false
            Single
                .fromCallable { mGeoDatabase.sendGeoLocationsDao().select(configuration.offlineMaxCount) }
                .flatMap { entities ->
                    Logger.d().addArg(entities.size).with(this@GeoRepositoryImpl).print()
                    if (entities.isNotEmpty()) {
                        val locations = entities.map { SendGeoLocationFactory.create(it) }
                        val coords = CoordsFactory.createCoords(mGeoExtensions, false, * locations.toTypedArray())
                        mSocketClient
                            .send(SocketData(coords.toByteArray()))
                            .map { isSended -> Pair(entities, isSended) }
                    } else {
                        Single.just(Pair(entities, false))
                    }
                }
                .doOnSuccess { (entities, isSended) ->
                    Logger.d().addArg(entities.size).addArg(isSended).with(this@GeoRepositoryImpl).print()
                    isLastSended = isSended
                    if (isSended) {
                        removeSendGeoLocations(entities)
                    }
                }
                .delay( if (count > 0) configuration.timeoutOnSendOffline else 100, TimeUnit.MILLISECONDS)
                .repeatUntil {
                    val rowCount = mGeoDatabase.sendGeoLocationsDao().getRowCount()
                    Logger.d("repeat").addArg(rowCount).addArg(isLastSended).with(this@GeoRepositoryImpl).print()
                    !isLastSended || rowCount == 0
                }
                .ignoreElements()
        }
        .doOnComplete {
            Logger.d("Complete").with(this@GeoRepositoryImpl).print()
        }

    override fun clear(): Completable = Completable
        .fromCallable {
            Logger.d().with(this@GeoRepositoryImpl).print()
            synchronized(mGeoDatabase.lock) {
                mGeoDatabase
                    .geosDao()
                    .deleteAll()
            }
        }
        .doOnComplete {
            Logger.d("Complete").with(this@GeoRepositoryImpl).print()
        }

    override fun sendGeo(geo: Geo): Completable = Single
        .fromCallable {
            Logger.d().addArg(geo).with(this@GeoRepositoryImpl).print()
            SocketData(CoordsFactory.createCoords(true, geo).toByteArray())
        }
        .flatMap { mSocketClient.send(it) }
        .doOnSuccess { isSended ->
            Logger.d().addArg(isSended).with(this@GeoRepositoryImpl).print()
            if (!isSended) {
                synchronized(mGeoDatabase.lock) {
                    mGeoDatabase
                        .geosDao()
                        .insert(GeoEntity(geo = geo))
                }
            }
        }
        .ignoreElement()
        .doOnComplete {
            Logger.d("Complete").with(this@GeoRepositoryImpl).print()
        }

    private fun removeSendGeoLocations(entities: List<SendGeoLocationEntity>) {
        Logger.d().addArg(entities.size).print()
        synchronized(mGeoDatabase.lock) {
            entities.forEach {
                mGeoDatabase
                    .sendGeoLocationsDao()
                    .delete(it)
            }
        }
    }

    private fun realSend(isOnline: Boolean, entities: List<GeoEntity>) {
        Logger.d().addArg(isOnline).addArg(entities.size).print()
        val coords = CoordsFactory.createCoords(isOnline, * entities.map { it.geo }.toTypedArray())
        val isSuccess = mSocketClient
            .send(SocketData(coords.toByteArray()))
            .blockingGet()
        if (isSuccess) {
            synchronized(mGeoDatabase.lock) {
                entities.forEach {
                    mGeoDatabase
                        .geosDao()
                        .delete(it)
                }
            }
        }
    }
}