package org.findmykids.geo.data.repository.storage.configuration

import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.db.GeoDatabase
import org.findmykids.geo.data.db.model.ConfigurationEntity
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Configurations
import javax.inject.Inject


internal class ConfigurationRepositoryImpl @Inject constructor(
    private val mGeoDatabase: GeoDatabase
) : ConfigurationRepository {

    private var mConfigurations: Configurations? = null


    override fun toString(): String = ""

    override fun get(): Single<Configurations> = Single
        .fromCallable {
            Logger.d().with(this@ConfigurationRepositoryImpl).print()
            if (mConfigurations == null) {
                val entities: List<ConfigurationEntity> = synchronized(mGeoDatabase.lock) {
                    try {
                        mGeoDatabase
                            .configurationsDao()
                            .selectAll()
                    } catch (e: Exception) {
                        Logger.e(e).with(this@ConfigurationRepositoryImpl).print()
                        mGeoDatabase
                            .configurationsDao()
                            .deleteAll()
                        listOf()
                    }
                }
                mConfigurations = Configurations(
                    entities.get<Configuration.ActivityConfiguration>() ?: Configuration.ActivityConfiguration(),
                    entities.get<Configuration.TimerConfiguration>() ?: Configuration.TimerConfiguration(),
                    entities.get<Configuration.PassiveConfiguration>() ?: Configuration.PassiveConfiguration(),
                    entities.get<Configuration.StationConfiguration>() ?: Configuration.StationConfiguration(),
                    entities.get<Configuration.LocationDataConfiguration>() ?: Configuration.LocationDataConfiguration(),
                    entities.get<Configuration.GpsDataConfiguration>() ?: Configuration.GpsDataConfiguration(),
                    entities.get<Configuration.LbsDataConfiguration>() ?: Configuration.LbsDataConfiguration(),
                    entities.get<Configuration.WifiDataConfiguration>() ?: Configuration.WifiDataConfiguration(),
                    entities.get<Configuration.SensorsDataConfiguration>() ?: Configuration.SensorsDataConfiguration(),
                    entities.get<Configuration.RemoteDataConfiguration>() ?: Configuration.RemoteDataConfiguration(),
                    entities.get<Configuration.TimeoutDataConfiguration>() ?: Configuration.TimeoutDataConfiguration(),
                    entities.get<Configuration.ActivityDataConfiguration>() ?: Configuration.ActivityDataConfiguration(),
                    entities.get<Configuration.BatteryDataConfiguration>() ?: Configuration.BatteryDataConfiguration(),
                    entities.get<Configuration.LocationLiveConfiguration>() ?: Configuration.LocationLiveConfiguration(),
                    entities.get<Configuration.LocatorLiveConfiguration>() ?: Configuration.LocatorLiveConfiguration(),
                    entities.get<Configuration.GeoStorageConfiguration>() ?: Configuration.GeoStorageConfiguration(),
                    entities.get<Configuration.ErrorStorageConfiguration>() ?: Configuration.ErrorStorageConfiguration()
                )
            }
            mConfigurations!!
        }
        .doOnSuccess {
            Logger.d().setResult(it).with(this@ConfigurationRepositoryImpl).print()
        }

    override fun set(configurations: Configurations): Completable = Completable
        .fromCallable {
            Logger.d().addArg(configurations).with(this@ConfigurationRepositoryImpl).print()
            synchronized(mGeoDatabase.lock) {
                mConfigurations = null
                mGeoDatabase.configurationsDao().apply {
                    deleteAll()
                    insert(ConfigurationEntity(configuration = configurations.activityConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.timerConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.passiveConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.stationConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.locationDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.activityDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.batteryDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.gpsDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.lbsDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.wifiDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.sensorsDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.socketDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.timeoutDataConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.locationLiveConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.locatorLiveConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.geoStorageConfiguration))
                    insert(ConfigurationEntity(configuration = configurations.errorStorageConfiguration))
                }
            }
        }
        .doOnComplete {
            Logger.d("Complete").with(this@ConfigurationRepositoryImpl).print()
        }

    override fun clear(): Completable = Completable
        .fromCallable {
            Logger.d().with(this@ConfigurationRepositoryImpl).print()
            synchronized(mGeoDatabase.lock) {
                mConfigurations = null
                mGeoDatabase
                    .configurationsDao()
                    .deleteAll()
            }
        }
        .doOnComplete {
            Logger.d("Complete").with(this@ConfigurationRepositoryImpl).print()
        }


    private inline fun <reified T> List<ConfigurationEntity>.get(): T? = this
        .firstOrNull { it.configuration is T }?.configuration as? T
}