package org.findmykids.geo.data.repository.live.sensors

import android.content.Context
import android.hardware.Sensor
import android.hardware.SensorEvent
import android.hardware.SensorEventListener
import android.hardware.SensorManager
import android.opengl.Matrix
import android.os.Handler
import android.os.HandlerThread
import androidx.core.content.ContextCompat
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import javax.inject.Inject


internal class SensorsRepositoryImpl @Inject constructor(
    private val mContext: Context
) : SensorsRepository(), SensorEventListener {

    private val mSensors = mutableListOf<Sensor>()
    private val mRotationMatrixInv = FloatArray(16)
    private var mCounter = 0L


    override fun toString(): String = ""

    override fun onAccuracyChanged(sensor: Sensor?, accuracy: Int) {
    }

    override fun onSensorChanged(event: SensorEvent) {
        if (mCounter++ % 500 == 0L) {
            Logger.d().print()
        }
        when (event.sensor.type) {
            Sensor.TYPE_LINEAR_ACCELERATION -> {
                val linearAcceleration = FloatArray(4)
                val absAcceleration = FloatArray(4)
                System.arraycopy(event.values, 0, linearAcceleration, 0, event.values.size)
                Matrix.multiplyMV(absAcceleration, 0, mRotationMatrixInv, 0, linearAcceleration, 0)
                sendEvent(SensorsEvent(SensorsFactory.create(absAcceleration)))
            }
            Sensor.TYPE_ROTATION_VECTOR -> {
                val rotationMatrix = FloatArray(16)
                SensorManager.getRotationMatrixFromVector(rotationMatrix, event.values)
                Matrix.invertM(mRotationMatrixInv, 0, rotationMatrix, 0)
            }
        }
    }

    override fun start(configuration: Configuration.SensorsDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        val sensorManager = ContextCompat.getSystemService(mContext, SensorManager::class.java)!!
        for (sensorType in SENSOR_TYPES) {
            val sensor = sensorManager.getDefaultSensor(sensorType) ?: continue
            mSensors.add(sensor)
        }
        if (mSensors.size != SENSOR_TYPES.size) {
            sendError(GeoException.SensorNoValidSensors())
        } else {
            mSensors.forEach { sensor ->
                sensorManager.registerListener(this, sensor, configuration.samplingPeriodUs, Handler(handlerThread.looper))
            }
        }
    }

    override fun stop() {
        Logger.i().print()
        val sensorManager = ContextCompat.getSystemService(mContext, SensorManager::class.java)!!
        mSensors.forEach {
            sensorManager.unregisterListener(this, it)
        }
        mSensors.clear()
    }


    companion object {
        private val SENSOR_TYPES = intArrayOf(Sensor.TYPE_LINEAR_ACCELERATION, Sensor.TYPE_ROTATION_VECTOR)
    }
}