@file:Suppress("DEPRECATION")

package org.findmykids.geo.data.repository.live.gps

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.content.Context
import android.content.pm.PackageManager
import android.location.GnssStatus
import android.location.GpsStatus
import android.location.LocationManager
import android.os.Build
import android.os.Handler
import android.os.HandlerThread
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.LocationManagerUtil
import org.findmykids.geo.data.model.Configuration
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class GpsRepositoryImpl @Inject constructor(
    private val mContext: Context
) : GpsRepository() {

    private val mLocationManager = ContextCompat.getSystemService(mContext, LocationManager::class.java)!!

    private var mDisposable: Disposable? = null


    private val mGnssStatusCallback: GnssStatus.Callback by lazy {
        @RequiresApi(Build.VERSION_CODES.N)
        object : GnssStatus.Callback() {
            override fun onSatelliteStatusChanged(status: GnssStatus) {
                Logger.d().with(this@GpsRepositoryImpl).print()
                sendEvent(GpsEvent(GpsInfoFactory.createFromGnssStatus(status)))
            }
        }
    }

    @Suppress("DEPRECATION", "MissingPermission")
    private val mGpsStatusListener: GpsStatus.Listener by lazy {
        GpsStatus.Listener {
            Logger.d().with(this@GpsRepositoryImpl).print()
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M ||
                mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED ||
                mContext.checkSelfPermission(ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED
            ) {
                sendEvent(GpsEvent(GpsInfoFactory.createFromGpsStatus(mLocationManager.getGpsStatus(null)!!)))
            }
        }
    }


    override fun toString(): String = ""

    override fun start(configuration: Configuration.GpsDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&
            mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
        ) {
            sendError(GeoException.GpsNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION))
            restart(configuration, handlerThread)
        } else if (LocationManagerUtil.isGpsEnabled(mLocationManager)) {
            Completable
                .fromCallable {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                        mLocationManager.registerGnssStatusCallback(mGnssStatusCallback, Handler(handlerThread.looper))
                    } else {
                        @Suppress("DEPRECATION")
                        mLocationManager.addGpsStatusListener(mGpsStatusListener)
                    }
                }
                .subscribeOn(AndroidSchedulers.mainThread())
                .subscribe()
        } else {
            sendError(GeoException.GpsLocationManagerGpsIsNotEnabled())
            restart(configuration, handlerThread)
        }
    }

    override fun stop() {
        Logger.i().print()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            mLocationManager.unregisterGnssStatusCallback(mGnssStatusCallback)
        } else {
            @Suppress("DEPRECATION")
            mLocationManager.removeGpsStatusListener(mGpsStatusListener)
        }
        mDisposable?.dispose()
        mDisposable = null
    }


    private fun restart(configuration: Configuration.GpsDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mDisposable = Completable
            .timer(configuration.restartDelay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                mDisposable = null
                Logger.d("restart").with(this@GpsRepositoryImpl).print()
                if (isStarted()) {
                    start(configuration, handlerThread)
                }
            }, {
                mDisposable = null
                sendError(GeoException.GpsRestart(it))
                restart(configuration, handlerThread)
            })
    }
}