package org.findmykids.geo.presentation.session

import android.annotation.SuppressLint
import android.app.Service
import android.content.Intent
import android.content.pm.PackageManager
import android.os.IBinder
import android.os.PowerManager
import android.os.PowerManager.WakeLock
import android.os.Process
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.ReplaySubject
import org.findmykids.geo.BuildConfig
import org.findmykids.geo.R
import org.findmykids.geo.api.extensions.SessionExtensions
import org.findmykids.geo.common.di.DIScopeHolder
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.NotificationUtil
import org.findmykids.geo.domain.factory.CommandFactory
import org.findmykids.geo.domain.model.Command
import org.findmykids.geo.presentation.session.input.CommandFrontController
import org.findmykids.geo.presentation.session.live.LiveFacade
import org.findmykids.geo.presentation.session.output.EventFrontController
import java.util.concurrent.TimeUnit
import javax.inject.Inject


/**
 * Главный сервис который дердит сессию, пока жив сервис сессия считается рабочей
 */
internal class SessionService : Service() {
    @Inject
    lateinit var mCommandFrontController: CommandFrontController

    @Inject
    lateinit var mLiveFacade: LiveFacade

    @Inject
    lateinit var mEventFrontController: EventFrontController

    @Inject
    lateinit var mSessionExtensions: SessionExtensions

    @Inject
    lateinit var mStartedJobs: StartedJobs

    @Volatile
    private var mIsForeground = false

    @Volatile
    private var mWakeLock: WakeLock? = null
    private var mQueueDisposable: Disposable? = null

    @Volatile
    private var mQueueSize = 0
    private val mQueue: ReplaySubject<Command> = ReplaySubject.create()

    private var mKillProcess: Disposable? = null


    override fun toString(): String = ""

    override fun onCreate() {
        Logger.i().print()
        super.onCreate()
        // set foreground
        if (!mIsForeground) {
            setForeground()
        }
        // not killing process
        mKillProcess?.dispose()
        mKillProcess = null
        // set wakelock
        if (mWakeLock == null) {
            setWakelock()
        }
    }

    override fun onDestroy() {
        Logger.i().print()
        super.onDestroy()
        // release queue
        mQueueDisposable?.let {
            if (!it.isDisposed) {
                it.dispose()
            }
        }
        mQueueDisposable = null
        // di close
        DIScopeHolder.clearSessionScope()
        // stop process
        mStartedJobs.stopJobs()
        mKillProcess = Completable
            .timer(10, TimeUnit.SECONDS)
            .subscribe {
                Logger.d("killing process").print()
                Process.killProcess(Process.myPid())
            }
        // release wakelock
        if (mWakeLock != null && mWakeLock!!.isHeld) {
            mWakeLock!!.release()
        }
        mWakeLock = null
        // release foreground
        stopForeground(true)
        mIsForeground = false
    }

    override fun onStartCommand(intent: Intent, flags: Int, startId: Int): Int {
        Logger.i().print()
        // set foreground
        if (!mIsForeground) {
            setForeground()
        }
        // set wakelock
        if (mWakeLock == null) {
            setWakelock()
        }
        // di
        DIScopeHolder.createSessionScope().inject(this)
        // init
        if (mQueueDisposable == null) {
            initQueue()
        }
        mQueue.onNext(CommandFactory.createInnerCommand(intent))
        return START_REDELIVER_INTENT
    }

    override fun onBind(intent: Intent?): IBinder? = null


    private fun setForeground() {
        Logger.d().print()
        mIsForeground = true
        val bundle = baseContext.packageManager.getApplicationInfo(baseContext.packageName, PackageManager.GET_META_DATA).metaData
        val title = bundle.getString(baseContext.getString(R.string.geo_notification_title))
        val description = bundle.getString(baseContext.getString(R.string.geo_notification_description))
        val icon = bundle.getInt(baseContext.getString(R.string.geo_notification_icon))
        if (title == null || description == null || icon == 0) {
            throw RuntimeException("Notification metadata not found in manifest")
        }
        startForeground(NOTIFICATION_ID, NotificationUtil.create(this, title, description, icon))
    }

    @SuppressLint("WakelockTimeout")
    private fun setWakelock() {
        Logger.d().print()
        try {
            mWakeLock = ContextCompat
                .getSystemService(this, PowerManager::class.java)!!
                .newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, SessionService::class.java.simpleName)
            if (!mWakeLock!!.isHeld) {
                mWakeLock!!.acquire()
            }
        } catch (e: Exception) {
            Logger.e(e).print()
            mWakeLock = null
        }
    }

    /**
     * Главная часть гео сервиса. представлеяет из себя две очереди.
     * Первая принимает на вход команды  {@see org.findmykids.geo.domain.model.Command} и либо запускает следующую очередь либо завершает ее.
     * Вторая очередь при запуске подписывается на события {@see org.findmykids.geo.domain.model.InnerEvent} и мониторит их. \
     * В зависимости от события принимает решение о завершении сессии.
     */
    private fun initQueue() {
        Logger.d().print()
        mSessionExtensions.onSessionBegin()
        mQueueDisposable = mQueue
            .doOnNext { command ->
                Logger.d("doOnNext").addArg(command).with(this@SessionService).print()
                if (command.isMain) {
                    mQueueSize++
                }
                mSessionExtensions.onNewCommand(command)
            }
            .subscribeOn(AndroidSchedulers.mainThread())
            .observeOn(Schedulers.single())
            .concatMap { command ->
                // обработка комманд
                Logger.d("concatMap").addArg(command).with(this@SessionService).print()
                mCommandFrontController
                    .add(command)
                    .map {(isActivateLive, left) ->
                        Logger.d("map").addArg(isActivateLive).addArg(left).with(this@SessionService).print()
                        if (left == 0 && command.isMain) {
                            mQueueSize--
                        }
                        isActivateLive
                    }
            }
            .distinctUntilChanged()
            .switchMap { isActivateLive ->
                Logger.d("switchMap").addArg(isActivateLive).with(this@SessionService).print()
                if (isActivateLive) {
                    mLiveFacade.start()
                } else {
                    mLiveFacade.stop()
                }
            }
            .concatMapSingle { innerEvent ->
                // принятие решения о завершении сессии
                Logger.d("concatMapSingle").addArg(innerEvent).with(this@SessionService).print()
                mEventFrontController.isNeedStopService(innerEvent)
            }
            .takeUntil { isNeedStopService ->
                Logger.d("takeUntil").addArg(isNeedStopService).addArg(mQueueSize).with(this@SessionService).print()
                if (isNeedStopService && (mQueueSize == 0 || mQueueSize > MAX_QUEUE_SIZE)){
                    Logger.d("Stop").with(this@SessionService).print()
                    true
                } else {
                    false
                }
            }
            .subscribeOn(Schedulers.single())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe({
                Logger.d("onNext").with(this@SessionService).print()
            }, { throwable ->
                Logger.e(throwable).with(this@SessionService).print()
                if (BuildConfig.DEBUG) {
                    throw throwable
                } else {
                    stopSelf()
                }
            }, {
                Logger.i("onComplete").with(this@SessionService).print()
                mSessionExtensions.onSessionEnd()
                stopSelf()
            })
    }


    companion object {
        /**
         * На некоторых телефонах (в частности Huawei) подписка на геолокацию, аларм менеджеры, активити сервис и т.д. происходит очень долго
         * поэтому на Tasks.await стоит ограничение в 10 секунд, но изза странной работы потоков в бекграунде (в частности всекого рода таймеры работают
         * дольше обычного, иногда кратно) этот await может висеть и 15 минут. Когда это происходит очередь комманд накапливается и отрабатывает крайне долго.
         * Что бы избежать зависания сессии будем отрубать сессии у которых очередь стала больше {@link MAX_QUEUE_SIZE}
         */
        private const val MAX_QUEUE_SIZE = 5
        private const val NOTIFICATION_ID = 425211
        const val ACTION_GEO = "${BuildConfig.LIBRARY_PACKAGE_NAME}.ACTION_GEO"
    }
}