package org.findmykids.geo.data.repository.trigger.passive.huaweifused

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import com.huawei.hmf.tasks.Tasks
import com.huawei.hms.location.LocationRequest
import com.huawei.hms.location.LocationServices
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.ManufactureUtil
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.trigger.passive.PassiveManager
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class PassiveHuaweiFusedManagerImpl @Inject constructor(
    private val mContext: Context
) : PassiveManager {

    override fun toString(): String = ""

    override fun isSubscribed(): Single<Boolean> = Single
        .fromCallable {
            Logger.d().with(this@PassiveHuaweiFusedManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE) != null
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveHuaweiFusedManagerImpl).print()
        }

    override fun subscribe(configuration: Configuration.PassiveConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@PassiveHuaweiFusedManagerImpl).print()
            if (!ManufactureUtil.isHuawei() || !ManufactureUtil.isEmuiVersion9_1_1orAfter()) {
                false
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&
                mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
                mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
            ) {
                Logger.e(GeoException.PassiveNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION))
                    .with(this@PassiveHuaweiFusedManagerImpl)
                    .print()
                false
            } else {
                try {
                    val client = LocationServices.getFusedLocationProviderClient(mContext)!!
                    // check is client works
                    Tasks.await(client.lastLocation, 10, TimeUnit.SECONDS)
                    getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                        Logger.d("has intent").with(this@PassiveHuaweiFusedManagerImpl).print()
                        val locationRequest = LocationRequest()
                            .setInterval(configuration.minTime)
                            .setSmallestDisplacement(configuration.minDistance)
                            .setPriority(configuration.priority)
                        Tasks.await(client.requestLocationUpdates(locationRequest, intent), 10, TimeUnit.SECONDS)
                        true
                    } ?: false
                } catch (e: Exception) {
                    Logger.e(GeoException.PassiveSubscription(e)).with(this@PassiveHuaweiFusedManagerImpl).print()
                    false
                }
            }
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveHuaweiFusedManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@PassiveHuaweiFusedManagerImpl).print()
            if (!ManufactureUtil.isHuawei() || !ManufactureUtil.isEmuiVersion9_1_1orAfter()) {
                true
            } else {
                getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
                    Logger.d("has intent").with(this@PassiveHuaweiFusedManagerImpl).print()
                    try {
                        Tasks.await(LocationServices.getFusedLocationProviderClient(mContext).removeLocationUpdates(intent), 10, TimeUnit.SECONDS)
                        true
                    } catch (e: Exception) {
                        // это ошибка получается исли мы пытаемся отписатся от того чего нет, но проверить есть ли подписка невозможно
                        if (e.message == "com.huawei.hms.common.ApiException: 10805: NO_MATCHED_INTENT") {
                            true
                        } else {
                            Logger.e(GeoException.PassiveUnSubscription(e)).with(this@PassiveHuaweiFusedManagerImpl).print()
                            false
                        }
                    } finally {
                        intent.cancel()
                    }
                } ?: true
            }
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@PassiveHuaweiFusedManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, PassiveHuaweiFusedReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_PASSIVE_HUAWEI_FUSED, intent, flags)
    }


    companion object {
        private const val REQUEST_PASSIVE_HUAWEI_FUSED = 6748604
    }
}