package org.findmykids.geo.data.repository.live.activity.huawei

import android.Manifest.permission.ACTIVITY_RECOGNITION
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager.PERMISSION_GRANTED
import android.os.Build
import android.os.HandlerThread
import com.huawei.hmf.tasks.Tasks
import com.huawei.hms.location.ActivityIdentification
import io.reactivex.Completable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.repository.live.activity.ActivityRepository
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class ActivityRepositoryHuaweiImpl @Inject constructor(
    private val mContext: Context
) : ActivityRepository() {

    private var mDisposable: Disposable? = null


    override fun toString(): String = ""

    override fun start(configuration: Configuration.ActivityDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().addArg(configuration).print()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q &&
            mContext.checkSelfPermission(ACTIVITY_RECOGNITION) != PERMISSION_GRANTED
        ) {
            sendError(GeoException.ActivityNoPermission(ACTIVITY_RECOGNITION))
            restart(configuration, handlerThread)
        } else {
            getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                Logger.d("has intent").print()
                try {
                    Tasks.await(
                        ActivityIdentification.getService(mContext).createActivityIdentificationUpdates(
                            configuration.detectionIntervalMillis,
                            intent
                        ), 10, TimeUnit.SECONDS
                    )
                    true
                } catch (e: Exception) {
                    sendError(GeoException.ActivityStart(e))
                    false
                }
            } ?: run {
                restart(configuration, handlerThread)
            }
        }
    }

    override fun stop() {
        Logger.d().print()
        mDisposable?.dispose()
        mDisposable = null
        getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
            Logger.d("has intent").print()
            try {
                Tasks.await(ActivityIdentification.getService(mContext).deleteActivityIdentificationUpdates(intent), 10, TimeUnit.SECONDS)
            } catch (e: Exception) {
                sendError(GeoException.ActivityStop(e))
            } finally {
                intent.cancel()
            }
        }
    }


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, ActivityHuaweiReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_ACTIVITY, intent, flags)
    }

    private fun restart(configuration: Configuration.ActivityDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mDisposable = Completable
            .timer(configuration.restartDelay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                mDisposable = null
                Logger.d("restart").with(this@ActivityRepositoryHuaweiImpl).print()
                if (isStarted()) {
                    start(configuration, handlerThread)
                }
            }, {
                mDisposable = null
                sendError(GeoException.ActivityRestart(it))
                restart(configuration, handlerThread)
            })
    }


    companion object {
        private const val REQUEST_ACTIVITY = 6748697
    }
}