package org.findmykids.geo.data.network

import android.os.Build
import com.google.gson.Gson
import com.google.gson.JsonObject
import io.reactivex.ObservableEmitter
import io.reactivex.Single
import io.reactivex.functions.Cancellable
import okhttp3.*
import okio.ByteString
import org.findmykids.geo.BuildConfig
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.network.factory.SocketDataFactory
import org.findmykids.geo.data.network.model.SocketCommand
import org.findmykids.geo.data.network.model.SocketData
import org.findmykids.geo.data.preferences.LocalPreferences
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class SocketClientImpl @Inject constructor(
    private val mLocalPreferences: LocalPreferences,
    private val mAppVersion: String
) : WebSocketListener(), SocketClient, Cancellable {

    private var mEmitter: ObservableEmitter<SocketData>? = null

    @Volatile
    private var mStatus = Status.DISCONNECTED

    private var mOkHttpClient: OkHttpClient = OkHttpClient()
    private var mWebSocket: WebSocket? = null

    @Volatile
    private var mConnectAfterDisconnecting: Boolean = false

    @Volatile
    private var mDisconnectAfterConnecting: Boolean = false


    override fun toString(): String = ""

    @Synchronized
    override fun subscribe(emitter: ObservableEmitter<SocketData>) {
        Logger.i().addArg(mStatus).print()
        mEmitter = emitter
        mDisconnectAfterConnecting = false
        when (mStatus) {
            Status.CONNECTED -> {
                mConnectAfterDisconnecting = false
                nothing()
            }
            Status.CONNECTING -> {
                mConnectAfterDisconnecting = false
                nothing()
            }
            Status.DISCONNECTING -> {
                mConnectAfterDisconnecting = true
                nothing()
            }
            Status.DISCONNECTED -> {
                mConnectAfterDisconnecting = false
                connecting()
            }
        }
        emitter.setCancellable(this)
    }

    @Synchronized
    override fun cancel() {
        Logger.i().addArg(mStatus).print()
        mEmitter = null
        mConnectAfterDisconnecting = false
        when (mStatus) {
            Status.CONNECTED -> {
                mDisconnectAfterConnecting = false
                disconnecting()
            }
            Status.CONNECTING -> {
                mDisconnectAfterConnecting = true
                nothing()
            }
            Status.DISCONNECTING -> {
                mDisconnectAfterConnecting = false
                nothing()
            }
            Status.DISCONNECTED -> {
                mDisconnectAfterConnecting = false
                nothing()
            }
        }
    }

    override fun send(socketData: SocketData): Single<Boolean> = Single
        .fromCallable {
            Logger.d().with(this@SocketClientImpl).addArg(mStatus).addArg(socketData).print()
            if (mStatus == Status.CONNECTED) {
                try {
                    mWebSocket?.send(ByteString.of(*socketData.toByteArray())) ?: false
                } catch (e: Exception) {
                    Logger.e(e).with(this@SocketClientImpl).print()
                    false
                }
            } else {
                false
            }
        }
        .flatMap {
            if (it) {
                Single
                    .timer(1, TimeUnit.SECONDS)
                    .map { mStatus == Status.CONNECTED }
            } else {
                Single.just(false)
            }
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@SocketClientImpl).print()
        }


    @Synchronized
    override fun onOpen(webSocket: WebSocket, response: Response) {
        Logger.d().addArg(mStatus).print()
        mStatus = Status.CONNECTED
        if (mDisconnectAfterConnecting) {
            mDisconnectAfterConnecting = false
            disconnecting()
        } else {
            mEmitter?.onNext(
                SocketData(
                    commandId = SocketCommand.START
                )
            )
        }
    }

    @Synchronized
    override fun onClosing(webSocket: WebSocket, code: Int, reason: String) {
        Logger.d().addArg(mStatus).addArg(code).addArg(reason).print()
    }

    @Synchronized
    override fun onClosed(webSocket: WebSocket, code: Int, reason: String) {
        Logger.d().addArg(mStatus).addArg(code).addArg(reason).print()
        mStatus = Status.DISCONNECTED
        mWebSocket = null
        if (mConnectAfterDisconnecting) {
            mConnectAfterDisconnecting = false
            connecting()
        } else if (code != NORMAL_CLOSURE_STATUS) {
            mEmitter?.onError(GeoException.SocketClose(code, reason))
        }
    }

    @Synchronized
    override fun onFailure(webSocket: WebSocket, t: Throwable, response: Response?) {
        Logger.e(t).addArg(mStatus).print()
        mStatus = Status.DISCONNECTED
        mWebSocket = null
        response?.body()?.string()?.let {
            val json = try {
                Gson().fromJson(it, JsonObject::class.java)
            } catch (e: Exception) {
                null
            }
            json?.let {
                if (json.has(ERROR_FIELD) && json.get(ERROR_FIELD).isJsonPrimitive) {
                    val jsonPrimitive = json.getAsJsonPrimitive(ERROR_FIELD)
                    if (jsonPrimitive.isString && jsonPrimitive.asString == "child_id is not equal to real id") {
                        mEmitter?.onError(GeoException.SocketInvalidUserId(json.get("child_id").asJsonPrimitive.asString.toCharArray()))
                        return
                    }
                }
            }
        }
        mEmitter?.onError(GeoException.Socket(t))
    }

    @Synchronized
    override fun onMessage(webSocket: WebSocket, bytes: ByteString) {
        Logger.d().addArg(mStatus).addArg(bytes).print()
        val socketData = try {
            SocketDataFactory.createSocketData(bytes.toByteArray())
        } catch (e: Exception) {
            Logger.e(e).print()
            null
        }
        socketData?.let {
            mEmitter?.onNext(it)
        }
    }

    @Synchronized
    override fun onMessage(webSocket: WebSocket, text: String) {
        Logger.d().addArg(mStatus).addArg(text).print()
        if (text == STOP_WORD) {
            disconnecting()
        }
    }


    private fun connecting() {
        Logger.d().addArg(mStatus).print()
        mStatus = Status.CONNECTING
        val request = Request.Builder()
            .url("${mLocalPreferences.getBaseUrl()!!}?child_id=//${String(mLocalPreferences.getUserId()!!)}")
            .addHeader("Geo-Version", BuildConfig.VERSION_NAME)
            .addHeader("Geo-Device", "android/" + Build.MANUFACTURER.replace(" ", "_") + "/" + Build.VERSION.SDK_INT)
            .addHeader("Geo-AccessToken", String(mLocalPreferences.getApiKey()!!))
            .addHeader("Geo-UserToken", String(mLocalPreferences.getUserToken()!!))
            .addHeader("Geo-AppVersion", mAppVersion)
            .addHeader("Geo-Session", "")
            .build()
        mWebSocket = mOkHttpClient.newWebSocket(request, this)
    }

    private fun disconnecting() {
        Logger.d().addArg(mStatus).print()
        mStatus = Status.DISCONNECTING
        mWebSocket?.close(NORMAL_CLOSURE_STATUS, STOP_WORD)
        mWebSocket = null
    }

    private fun nothing() {
        Logger.d().print()
    }


    enum class Status {
        CONNECTED, // подключен
        CONNECTING, // подключается
        DISCONNECTING, // отключается
        DISCONNECTED // отключен
    }


    companion object {
        private const val NORMAL_CLOSURE_STATUS = 1000
        private const val STOP_WORD = "goodbye"
        private const val ERROR_FIELD = "error"
    }
}