@file:Suppress("unused")

package org.findmykids.geo.api

import android.app.Activity
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.database.CursorIndexOutOfBoundsException
import android.net.Uri
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.work.*
import org.findmykids.geo._todo.GeoConverter
import org.findmykids.geo.api.listeners.GeoListener
import org.findmykids.geo.api.model.GeoPlatformState
import org.findmykids.geo.data.repository.trigger.task.TaskWorker
import org.findmykids.geo.data.repository.trigger.task.Tasks
import org.findmykids.geo.presentation.manage.AuthorityCheckProvider
import org.findmykids.geo.presentation.session.SessionService
import java.util.concurrent.TimeUnit


/**
 * Основной класс для управления гео платформы
 * Общение происходит через startService, startForegroundService и WorkManager в зависимости от условий
 */
object GeoPlatform {

    private var mGeoListener: GeoListener? = null
    private var mGeoServiceBroadcastReceiver = lazy {
        object : BroadcastReceiver() {
            private val mGeoConverter = GeoConverter()

            override fun onReceive(context: Context?, intent: Intent) {
//                mGeoListener?.onNewGeo(mGeoConverter.toGeo(intent.getStringExtra(SessionService.EXTRA_GEO)!!))
            }
        }
    }


    /**
     * Листенер на смену геолокации
     * Не работает, после изменений не дошли руки его доделать
     */
    @JvmStatic
    fun setListener(context: Context, geoListener: GeoListener?) {
        if (geoListener == null) {
            if (mGeoListener != null) {
                context.unregisterReceiver(mGeoServiceBroadcastReceiver.value)
                mGeoListener = null
            }
        } else {
            if (mGeoListener == null) {
                val intentFilter = IntentFilter(SessionService.ACTION_GEO)
                context.registerReceiver(mGeoServiceBroadcastReceiver.value, intentFilter)
            }
            mGeoListener = geoListener
        }
    }


    /**
     * Узнать состояние гео платформы
     */
    @JvmStatic
    @Synchronized
    fun isActivatedTo(context: Context, baseUrl: String, userId: CharArray): GeoPlatformState {
        var i = 0
        do {
            val cursor = try {
                context.contentResolver.query(
                    Uri.parse("content://" + GeoPlatformSettings.mAppId + "." + AuthorityCheckProvider.NAME),
                    null,
                    null,
                    arrayOf(baseUrl, String(userId)),
                    null
                )
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
            if (cursor == null) {
                ping(context)
                try {
                    Thread.sleep(200)
                } catch (e: InterruptedException) {
                    e.printStackTrace()
                }
                i++
            } else {
                try {
                    if (cursor.moveToFirst()) {
                        return GeoPlatformState.getState(cursor.getInt(0))
                    }
                } catch (e: CursorIndexOutOfBoundsException) {
                    e.printStackTrace()
                } finally {
                    try {
                        cursor.close()
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }
        } while (i < 3)
        return GeoPlatformState.ERROR
    }

    /**
     * Активация гео платформы
     */
    @JvmStatic
    fun activate(context: Context, baseUrl: String, apiKey: CharArray, userToken: CharArray, userId: CharArray) {
        when {
            context is Activity -> execute(context, getIntent(context, Tasks.ACTIVATE).apply {
                putExtra(Tasks.BASE_URL, baseUrl)
                putExtra(Tasks.API_KEY, apiKey)
                putExtra(Tasks.USER_TOKEN, userToken)
                putExtra(Tasks.USER_ID, userId)
            })
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.O -> execute(context, getData(Tasks.ACTIVATE).apply {
                putString(Tasks.BASE_URL, baseUrl)
                putString(Tasks.API_KEY, String(apiKey))
                putString(Tasks.USER_TOKEN, String(userToken))
                putString(Tasks.USER_ID, String(userId))
            }.build())
            else -> context.startService(getIntent(context, Tasks.ACTIVATE).apply {
                putExtra(Tasks.BASE_URL, baseUrl)
                putExtra(Tasks.API_KEY, apiKey)
                putExtra(Tasks.USER_TOKEN, userToken)
                putExtra(Tasks.USER_ID, userId)
            })
        }
    }

    /**
     * деактивация геоплатформы
     */
    @JvmStatic
    fun deactivate(context: Context) {
        when {
            context is Activity -> execute(context, getIntent(context, Tasks.DEACTIVATE))
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.O -> execute(context, getData(Tasks.DEACTIVATE).build())
            else -> context.startService(getIntent(context, Tasks.DEACTIVATE))
        }
    }


    /**
     * Запустить геопалторму
     */
    @JvmStatic
    fun ping(context: Context) {
        when {
            context is Activity -> execute(context, getIntent(context, Tasks.PING))
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.O -> execute(context, getData(Tasks.PING).build())
            else -> context.startService(getIntent(context, Tasks.PING))
        }
    }


    /**
     * Запрос координаты и посылает ее на сервер. не используется, поэтому может не работать
     */
    @JvmStatic
    fun locationRequest(context: Context) {
        when {
            context is Activity -> execute(context, getIntent(context, Tasks.REQUEST))
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.O -> execute(context, getData(Tasks.REQUEST).build())
            else -> context.startService(getIntent(context, Tasks.REQUEST))
        }
    }


    /**
     * Включить или отключить реалтайм
     */
    @JvmStatic
    fun setRealtime(context: Context, enable: Boolean) {
        when {
            context is Activity -> execute(context, getIntent(context, Tasks.REAL_TIME).apply {
                putExtra(Tasks.REAL_TIME_ENABLED, enable)
            })
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.O -> execute(context, getData(Tasks.REAL_TIME).apply {
                putBoolean(Tasks.REAL_TIME_ENABLED, enable)
            }.build())
            else -> context.startService(getIntent(context, Tasks.REAL_TIME).apply {
                putExtra(Tasks.REAL_TIME_ENABLED, enable)
            })
        }
    }


    private fun getIntent(context: Context, task: Tasks) =
        Intent(context, SessionService::class.java).apply {
            action = task.name
        }

    private fun execute(context: Context, intent: Intent) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            context.startForegroundService(intent)
        } else {
            context.startService(intent)
        }
    }


    @RequiresApi(Build.VERSION_CODES.O)
    private fun getData(task: Tasks) =
        Data.Builder().putString(TaskWorker.TASK, task.name)

    @RequiresApi(Build.VERSION_CODES.O)
    private fun execute(context: Context, data: Data) {
        val constraints = Constraints.Builder()
            .setRequiredNetworkType(NetworkType.NOT_REQUIRED)
            .setTriggerContentUpdateDelay(TimeUnit.SECONDS.toMillis(2), TimeUnit.MILLISECONDS)
            .setTriggerContentMaxDelay(TimeUnit.SECONDS.toMillis(2), TimeUnit.MILLISECONDS)
            .build()
        val work = OneTimeWorkRequestBuilder<TaskWorker>()
            .setConstraints(constraints)
            .setBackoffCriteria(BackoffPolicy.LINEAR, TimeUnit.SECONDS.toMillis(2), TimeUnit.MILLISECONDS)
            .setInitialDelay(TimeUnit.SECONDS.toMillis(1), TimeUnit.MILLISECONDS)
            .setInputData(data)
            .build()
        WorkManager.getInstance(context).enqueue(work)
    }
}