package org.findmykids.geo._todo

import androidx.room.TypeConverter
import com.google.gson.Gson
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import com.google.gson.reflect.TypeToken
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.data.model.*
import java.util.*


//TODO яндекс
internal class GeoConverter {
    @TypeConverter
    fun toGeo(jsonString: String): Geo {
        val json = GSON.fromJson(jsonString, JsonObject::class.java)
        val create = Date(json.get(FIELD_CREATE).asLong)
        val session = json.get(FIELD_SESSION).asJsonObject.let { sessionJson ->
            Session(
                sessionJson.get(FIELD_SESSION_INDEX).asLong,
                Date(sessionJson.get(FIELD_SESSION_CREATE).asLong),
                sessionJson.get(FIELD_SESSION_IS_REALTIME).asBoolean,
                sessionJson.get(FIELD_SESSION_COMMANDS).asJsonArray.map {
                    when (val type = it.asJsonObject.get(FIELD_SESSION_COMMAND_TYPE).asInt) {
                        CommandType.ACTIVATE.value -> CommandType.ACTIVATE
                        CommandType.DEACTIVATE.value -> CommandType.DEACTIVATE
                        CommandType.PING.value -> CommandType.PING
                        CommandType.REQUEST.value -> CommandType.REQUEST
                        CommandType.TIMER.value -> CommandType.TIMER
                        CommandType.BOOT.value -> CommandType.BOOT
                        CommandType.ACTIVITY.value -> CommandType.ACTIVITY
                        CommandType.ZONE.value -> CommandType.ZONE
                        CommandType.STATION.value -> CommandType.STATION
                        CommandType.PASSIVE.value -> CommandType.PASSIVE
                        CommandType.REALTIME.value -> CommandType.REALTIME
                        else -> throw GeoException.InvalidCommandType(type)
                    }
                }
            )
        }
        val extensions = json.get(FIELD_EXTENSIONS).asJsonArray.let { propertiesJson ->
            mutableMapOf<String, String>().apply {
                propertiesJson.forEach {
                    val key = it.asJsonObject.get(FIELD_EXTENSION_KEY).asString
                    val value = it.asJsonObject.get(FIELD_EXTENSION_VALUE).asString
                    put(key, value)
                }
            }
        }
        return when (val type = json.get(FIELD_TYPE).asInt) {
            Geo.Type.LBS_LOCATOR_LOCATION.value -> Geo.LbsLocatorLocationGeo(
                GSON.fromJson(json.get(FIELD_1).asString, object : TypeToken<ArrayList<Lbs>>() {}.type),
                GSON.fromJson(json.get(FIELD_2).asString, object : TypeToken<ArrayList<WifiFull>>() {}.type),
                GSON.fromJson(json.get(FIELD_3).asString, LocatorLocation::class.java),
                session,
                create,
                extensions
            )
            Geo.Type.LBS_LOCATOR_ERROR.value -> Geo.LbsLocatorErrorGeo(
                GSON.fromJson(json.get(FIELD_1).asString, object : TypeToken<ArrayList<Lbs>>() {}.type),
                GSON.fromJson(json.get(FIELD_2).asString, object : TypeToken<ArrayList<WifiFull>>() {}.type),
                GSON.fromJson(json.get(FIELD_3).asString, String::class.java),
                session,
                create,
                extensions
            )
            Geo.Type.WIFI_LOCATOR_LOCATION.value -> Geo.WifiLocatorLocationGeo(
                GSON.fromJson(json.get(FIELD_1).asString, object : TypeToken<ArrayList<WifiFull>>() {}.type),
                GSON.fromJson(json.get(FIELD_2).asString, object : TypeToken<ArrayList<Lbs>>() {}.type),
                GSON.fromJson(json.get(FIELD_3).asString, LocatorLocation::class.java),
                session,
                create,
                extensions
            )
            Geo.Type.WIFI_LOCATOR_ERROR.value -> Geo.WifiLocatorErrorGeo(
                GSON.fromJson(json.get(FIELD_1).asString, object : TypeToken<ArrayList<WifiFull>>() {}.type),
                GSON.fromJson(json.get(FIELD_2).asString, object : TypeToken<ArrayList<Lbs>>() {}.type),
                GSON.fromJson(json.get(FIELD_3).asString, String::class.java),
                session,
                create,
                extensions
            )
            else -> throw GeoException.InvalidGeoType(type)
        }
    }

    @TypeConverter
    fun fromGeo(geo: Geo): String {
        val json = JsonObject()

        json.addProperty(FIELD_CREATE, geo.create.time)

        json.addProperty(FIELD_TYPE, geo.type)

        when (geo) {
            is Geo.LbsLocatorLocationGeo -> {
                json.addProperty(FIELD_1, GSON.toJson(geo.lbsList))
                json.addProperty(FIELD_2, GSON.toJson(geo.wifiFullList))
                json.addProperty(FIELD_3, GSON.toJson(geo.locatorLocation))
            }
            is Geo.LbsLocatorErrorGeo -> {
                json.addProperty(FIELD_1, GSON.toJson(geo.lbsList))
                json.addProperty(FIELD_2, GSON.toJson(geo.wifiFullList))
                json.addProperty(FIELD_3, GSON.toJson(geo.locatorError))
            }
            is Geo.WifiLocatorLocationGeo -> {
                json.addProperty(FIELD_1, GSON.toJson(geo.wifiFullList))
                json.addProperty(FIELD_2, GSON.toJson(geo.lbsList))
                json.addProperty(FIELD_3, GSON.toJson(geo.locatorLocation))
            }
            is Geo.WifiLocatorErrorGeo -> {
                json.addProperty(FIELD_1, GSON.toJson(geo.wifiFullList))
                json.addProperty(FIELD_2, GSON.toJson(geo.lbsList))
                json.addProperty(FIELD_3, GSON.toJson(geo.locatorError))
            }
        }

        json.add(FIELD_SESSION, JsonObject().apply {
            addProperty(FIELD_SESSION_INDEX, geo.session.index)
            addProperty(FIELD_SESSION_CREATE, geo.session.create.time)
            addProperty(FIELD_SESSION_IS_REALTIME, geo.session.isRealtime)
            add(FIELD_SESSION_COMMANDS, JsonArray().apply {
                geo.session.commandsTypes.forEach { command ->
                    add(JsonObject().also {
                        it.addProperty(FIELD_SESSION_COMMAND_TYPE, command.value)
                    })
                }
            })
        })

        json.add(FIELD_EXTENSIONS, JsonArray().apply {
            geo.extensions.map { (key, value) ->
                JsonObject().apply {
                    addProperty(FIELD_EXTENSION_KEY, key)
                    addProperty(FIELD_EXTENSION_VALUE, value)
                }
            }.forEach {
                add(it)
            }
        })

        return json.toString()
    }


    companion object {
        private const val FIELD_TYPE = "type"
        private const val FIELD_CREATE = "create"
        private const val FIELD_1 = "field1"
        private const val FIELD_2 = "field2"
        private const val FIELD_3 = "field3"
        private const val FIELD_SESSION = "session"
        private const val FIELD_SESSION_INDEX = "index"
        private const val FIELD_SESSION_CREATE = "create"
        private const val FIELD_SESSION_IS_REALTIME = "isRealtime"
        private const val FIELD_SESSION_COMMANDS = "commands"
        private const val FIELD_SESSION_COMMAND_TYPE = "commandType"
        private const val FIELD_EXTENSIONS = "extensions"
        private const val FIELD_EXTENSION_KEY = "extensionKey"
        private const val FIELD_EXTENSION_VALUE = "extensionValue"

        private val GSON = Gson()
    }
}