package org.findmykids.geo.domain.live.error

import io.reactivex.Observable
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Activity
import org.findmykids.geo.data.model.Battery
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Error
import org.findmykids.geo.data.repository.live.activity.ActivityRepository
import org.findmykids.geo.data.repository.live.battery.BatteryRepository
import org.findmykids.geo.data.repository.live.gps.GpsRepository
import org.findmykids.geo.data.repository.live.lbs.LbsRepository
import org.findmykids.geo.data.repository.live.location.LocationRepository
import org.findmykids.geo.data.repository.live.remote.RemoteRepository
import org.findmykids.geo.data.repository.live.sensors.SensorsRepository
import org.findmykids.geo.data.repository.live.timeout.TimeoutRepository
import org.findmykids.geo.data.repository.live.wifi.WifiRepository
import org.findmykids.geo.data.repository.storage.configuration.ConfigurationRepository
import org.findmykids.geo.data.repository.storage.currentSession.CurrentSessionRepository
import org.findmykids.geo.data.repository.storage.error.ErrorRepository
import org.findmykids.geo.domain.BaseInteractor
import org.findmykids.geo.domain.model.InnerEvent
import java.util.*
import javax.inject.Inject


internal class ErrorInteractorImpl @Inject constructor(
    private val mConfigurationRepository: ConfigurationRepository,
    private val mErrorRepository: ErrorRepository,
    private val mCurrentSessionRepository: CurrentSessionRepository,
    private val mActivityRepository: ActivityRepository,
    private val mBatteryRepository: BatteryRepository,
    private val mGpsRepository: GpsRepository,
    private val mLbsRepository: LbsRepository,
    private val mLocationRepository: LocationRepository,
//    private val mRemoteRepository: RemoteRepository,
    private val mSensorsRepository: SensorsRepository,
    private val mTimeoutRepository: TimeoutRepository,
    private val mWifiRepository: WifiRepository
) : BaseInteractor(), ErrorInteractor {

    private var mBattery: Battery? = null
    private var mActivity: Activity? = null

    /**
     * Ранее обработанные ошибки
     */
    private val mPreviousErrors = mutableListOf<Error>()


    override fun toString(): String = ""

    override fun update(): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .doOnSubscribe {
            Logger.d().with(this@ErrorInteractorImpl).print()
            mActivity = null
            mBattery = null
        }
        .flatMapObservable {
            // Подписываемся на получение ошибок
            Logger.d().with(this@ErrorInteractorImpl).print()
            Observable.mergeArray(
                mActivityRepository.observeErrors(),
                activityEmitter(it.activityDataConfiguration),
                mBatteryRepository.observeErrors(),
                batteryEmitter(it.batteryDataConfiguration),
                mGpsRepository.observeErrors(),
                mLbsRepository.observeErrors(),
                mLocationRepository.observeErrors(),
//                mRemoteRepository.observeErrors(),
                mSensorsRepository.observeErrors(),
                mTimeoutRepository.observeErrors(),
                mWifiRepository.observeErrors()
            )
        }
        .flatMapSingle { geoException ->
            Logger.d().addArg(geoException).with(this@ErrorInteractorImpl).print()
            mCurrentSessionRepository
                .getSession()
                .map { Error(Date(), it, mActivity, mBattery, geoException) }
        }
        .flatMap { error ->
            // Одинаковые ошибки не отправляются - иначе бэк нагнется
            Logger.d().addArg(error).with(this@ErrorInteractorImpl).print()
            val sameError = mPreviousErrors
                .firstOrNull { it.geoException.toString() == error.geoException.toString() }
            if (sameError == null) {
                mPreviousErrors.add(error)
                mErrorRepository
                    .send(error)
                    .andThen(Observable.just(InnerEvent.NewError(error)))
            } else {
                Observable.just(InnerEvent.NewError(error))
            }
        }


    private fun activityEmitter(configuration: Configuration.ActivityDataConfiguration): Observable<GeoException> = mActivityRepository
        .observeEvents(configuration)
        .flatMap { event ->
            Logger.d().addArg(event).with(this@ErrorInteractorImpl).print()
            mActivity = event.activity
            Observable.empty<GeoException>()
        }

    private fun batteryEmitter(configuration: Configuration.BatteryDataConfiguration): Observable<GeoException> = mBatteryRepository
        .observeEvents(configuration)
        .flatMap { event ->
            Logger.d().addArg(event).with(this@ErrorInteractorImpl).print()
            mBattery = event.battery
            Observable.empty<GeoException>()
        }
}