package org.findmykids.geo.data.repository.live.timeout

import android.os.HandlerThread
import io.reactivex.Completable
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class TimeoutRepositoryImpl @Inject constructor(
) : TimeoutRepository() {

    private var mTimerDisposable: Disposable? = null
    private var mCreateTime: Long = System.currentTimeMillis()
    private var mRestartDisposable: Disposable? = null


    override fun toString(): String = ""

    override fun start(configuration: Configuration.TimeoutDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        mRestartDisposable?.dispose()
        mRestartDisposable = null
        mTimerDisposable = Observable
            .interval(configuration.delay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                Logger.d("success").with(this@TimeoutRepositoryImpl).print()
                val workTime = System.currentTimeMillis() - mCreateTime
                sendEvent(TimeoutEvent(workTime))
            }, { throwable ->
                mTimerDisposable = null
                sendError(GeoException.TimeoutStart(throwable))
                restart(configuration, handlerThread)
            })
    }

    override fun stop() {
        Logger.i().print()
        mTimerDisposable?.dispose()
        mTimerDisposable = null
        mRestartDisposable?.dispose()
        mRestartDisposable = null
    }


    private fun restart(configuration: Configuration.TimeoutDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mRestartDisposable = Completable
            .timer(configuration.restartDelay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                mRestartDisposable = null
                Logger.d("restart").with(this@TimeoutRepositoryImpl).print()
                if (isStarted()) {
                    start(configuration, handlerThread)
                }
            }, {
                mRestartDisposable = null
                sendError(GeoException.TimeoutRestart(it))
                restart(configuration, handlerThread)
            })
    }
}