package org.findmykids.geo.data.network.factory

import com.google.gson.JsonArray
import com.google.gson.JsonObject
import geoproto.*
import geoproto.Activity
import geoproto.Lbs
import geoproto.Location
import geoproto.Wifi
import org.findmykids.geo._todo.Geo
import org.findmykids.geo.api.extensions.GeoExtensions
import org.findmykids.geo.data.model.*
import java.text.SimpleDateFormat
import java.util.*


internal object CoordFactory {
    private const val TS_PATTERN = "yyyy-MM-dd'T'HH:mm:ssZZZZZ"
    private const val SESSION_INDEX = "session_index"
    private const val DEFINE_INDEX = "define_index"
    private const val SEND_INDEX = "send_index"
    private const val SESSION_START = "session_start"
    private const val SEND_TIME = "send_time"
    private const val COMMANDS = "commands"
    private const val SEND_REASON = "send_reason"
    private const val ERROR = "error"
    private const val EMPTY_BATTERY_LEVEL_VALUE = -1


    // region
    fun createCoord(geo: Geo.LbsLocatorLocationGeo): Coord = Coord
        .newBuilder()
        .apply {
            geo.locatorLocation.let { location ->
                addLocations(
                    Location.newBuilder()
                        .setProvider(Provider.YANDEX)
                        .setLatitude(location.latitude)
                        .setLongitude(location.longitude)
                        .setAccuracy(location.accuracy.toLong())
                        .setAge(System.currentTimeMillis() - location.time)
                        .setAltitude(location.altitude.toFloat())
                        .build()
                )
            }
            geo.lbsList.forEach {
                addLbs(
                    Lbs.newBuilder()
                        .setCountryCode(it.countryCode)
                        .setOperatorId(it.operatorId)
                        .setCellId(it.cellId)
                        .setLac(it.lac)
                        .setSignalStrength(it.level)
                        .build()
                )
            }
            geo.wifiFullList.forEach {
                addWifi(
                    Wifi.newBuilder()
                        .setMac(it.bssid)
                        .setSignalStrength(it.level)
                        .setState(it.isConnected)
                        .setName(it.ssid)
                        .build()
                )
            }
            activityType = Activity.UNKNOWN
            if (geo.extensions.containsKey("isCharging")) {
                isCharging = geo.extensions["isCharging"]!!.toBoolean()
            }
            batteryLevel = if (geo.extensions.containsKey("level")) {
                geo.extensions["level"]!!.toInt()
            } else {
                EMPTY_BATTERY_LEVEL_VALUE
            }
            debug = JsonObject().apply {
                addProperty(SESSION_INDEX, geo.session.index)
                addProperty(SESSION_START, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(geo.session.create))
                addProperty(SEND_TIME, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(Date()))
                add(COMMANDS, JsonArray().also { reasons ->
                    geo.session.commandsTypes.forEach {
                        reasons.add(it.value)
                    }
                })
            }.toString()
        }
        .setOk()
        .setTs(geo.create)
        .setReason(geo.session)
        .build()

    fun createCoord(geo: Geo.LbsLocatorErrorGeo): Coord = Coord
        .newBuilder()
        .apply {
            geo.lbsList.forEach {
                addLbs(
                    Lbs.newBuilder()
                        .setCountryCode(it.countryCode)
                        .setOperatorId(it.operatorId)
                        .setCellId(it.cellId)
                        .setLac(it.lac)
                        .setSignalStrength(it.level)
                        .build()
                )
            }
            geo.wifiFullList.forEach {
                addWifi(
                    Wifi.newBuilder()
                        .setMac(it.bssid)
                        .setSignalStrength(it.level)
                        .setState(it.isConnected)
                        .setName(it.ssid)
                        .build()
                )
            }
            activityType = Activity.UNKNOWN
            if (geo.extensions.containsKey("isCharging")) {
                isCharging = geo.extensions["isCharging"]!!.toBoolean()
            }
            batteryLevel = if (geo.extensions.containsKey("level")) {
                geo.extensions["level"]!!.toInt()
            } else {
                EMPTY_BATTERY_LEVEL_VALUE
            }
            debug = JsonObject().apply {
                addProperty(SESSION_INDEX, geo.session.index)
                addProperty(SESSION_START, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(geo.session.create))
                addProperty(SEND_TIME, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(Date()))
                add(COMMANDS, JsonArray().also { reasons ->
                    geo.session.commandsTypes.forEach {
                        reasons.add(it.value)
                    }
                })
            }.toString()
        }
        .setOk()
        .setTs(geo.create)
        .setReason(geo.session)
        .build()

    fun createCoord(geo: Geo.WifiLocatorLocationGeo): Coord = Coord
        .newBuilder()
        .apply {
            geo.locatorLocation.let { location ->
                addLocations(
                    Location.newBuilder()
                        .setProvider(Provider.YANDEX)
                        .setLatitude(location.latitude)
                        .setLongitude(location.longitude)
                        .setAccuracy(location.accuracy.toLong())
                        .setAge(System.currentTimeMillis() - location.time)
                        .setAltitude(location.altitude.toFloat())
                        .build()
                )
            }
            geo.wifiFullList.forEach {
                addWifi(
                    Wifi.newBuilder()
                        .setMac(it.bssid)
                        .setSignalStrength(it.level)
                        .setState(it.isConnected)
                        .setName(it.ssid)
                        .build()
                )
            }
            geo.lbsList.forEach {
                addLbs(
                    Lbs.newBuilder()
                        .setCountryCode(it.countryCode)
                        .setOperatorId(it.operatorId)
                        .setCellId(it.cellId)
                        .setLac(it.lac)
                        .setSignalStrength(it.level)
                        .build()
                )
            }
            activityType = Activity.UNKNOWN
            if (geo.extensions.containsKey("isCharging")) {
                isCharging = geo.extensions["isCharging"]!!.toBoolean()
            }
            batteryLevel = if (geo.extensions.containsKey("level")) {
                geo.extensions["level"]!!.toInt()
            } else {
                EMPTY_BATTERY_LEVEL_VALUE
            }
            debug = JsonObject().apply {
                addProperty(SESSION_INDEX, geo.session.index)
                addProperty(SESSION_START, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(geo.session.create))
                addProperty(SEND_TIME, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(Date()))
                add(COMMANDS, JsonArray().also { reasons ->
                    geo.session.commandsTypes.forEach {
                        reasons.add(it.value)
                    }
                })
            }.toString()
        }
        .setOk()
        .setTs(geo.create)
        .setReason(geo.session)
        .build()

    fun createCoord(geo: Geo.WifiLocatorErrorGeo): Coord = Coord
        .newBuilder()
        .apply {
            geo.wifiFullList.forEach {
                addWifi(
                    Wifi.newBuilder()
                        .setMac(it.bssid)
                        .setSignalStrength(it.level)
                        .setState(it.isConnected)
                        .setName(it.ssid)
                        .build()
                )
            }
            geo.lbsList.forEach {
                addLbs(
                    Lbs.newBuilder()
                        .setCountryCode(it.countryCode)
                        .setOperatorId(it.operatorId)
                        .setCellId(it.cellId)
                        .setLac(it.lac)
                        .setSignalStrength(it.level)
                        .build()
                )
            }
            activityType = Activity.UNKNOWN
            if (geo.extensions.containsKey("isCharging")) {
                isCharging = geo.extensions["isCharging"]!!.toBoolean()
            }
            batteryLevel = if (geo.extensions.containsKey("level")) {
                geo.extensions["level"]!!.toInt()
            } else {
                EMPTY_BATTERY_LEVEL_VALUE
            }
            debug = JsonObject().apply {
                addProperty(SESSION_INDEX, geo.session.index)
                addProperty(SESSION_START, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(geo.session.create))
                addProperty(SEND_TIME, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(Date()))
                add(COMMANDS, JsonArray().also { reasons ->
                    geo.session.commandsTypes.forEach {
                        reasons.add(it.value)
                    }
                })
            }.toString()
        }
        .setOk()
        .setTs(geo.create)
        .setReason(geo.session)
        .build()
    // endregion


    fun createCoord(error: Error): Coord = Coord
        .newBuilder()
        .apply {
            failureReason = FailureReason.newBuilder()
                .setCode(FailureReasonCode.EMPTY)
                .setDescription(error.geoException.toString())
                .build()
            debug = JsonObject().apply {
                addProperty(SESSION_INDEX, error.session.index)
                addProperty(SESSION_START, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(error.session.create))
                addProperty(SEND_TIME, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(Date()))
                add(COMMANDS, JsonArray().also { reasons ->
                    error.session.commandsTypes.forEach {
                        reasons.add(it.value)
                    }
                })
                addProperty(ERROR, error.geoException.toString())
            }.toString()
        }
        .setTs(error.create)
        .setActivity(error.activity)
        .setBattery(error.battery)
        .setReason(error.session)
        .build()


    fun createCoord(sendGeoLocation: SendGeoLocation, geoExtensions: GeoExtensions): Coord = Coord
        .newBuilder()
        .apply {
            debug = JsonObject().apply {
                addProperty(SESSION_INDEX, sendGeoLocation.session.index)
                addProperty(DEFINE_INDEX, "${sendGeoLocation.locationDefineSessionIndex}/${sendGeoLocation.locationDefineGlobalIndex}")
                addProperty(SEND_INDEX, "${sendGeoLocation.sendSessionIndex}/${sendGeoLocation.sendGlobalIndex}")
                addProperty(SEND_REASON, sendGeoLocation.sendReason)
                addProperty(SESSION_START, SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(sendGeoLocation.session.create))
                addProperty(SEND_TIME, SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSZZZZZ", Locale.ENGLISH).format(Date()))
                add(COMMANDS, JsonArray().also { reasons ->
                    sendGeoLocation.session.commandsTypes.forEach {
                        reasons.add(it.value)
                    }
                })
                sendGeoLocation.calculatedLocation?.let { calculatedLocation ->
                    add("kalman", JsonObject().apply {
                        addProperty("latitude", calculatedLocation.latitude)
                        addProperty("longitude", calculatedLocation.longitude)
                        addProperty("speed", calculatedLocation.speed)
                        addProperty("altitude", calculatedLocation.altitude)
                    })
                }
            }.toString()
        }
        .setOk()
        .setTs(sendGeoLocation.create)
        .setLocation(sendGeoLocation.location)
        .setBattery(sendGeoLocation.battery)
        .setActivity(sendGeoLocation.activity)
        .setGps(sendGeoLocation.gpsInfo)
        .setWifi(sendGeoLocation.wifi)
        .setReason(sendGeoLocation.session)
        .setExtensions(sendGeoLocation, geoExtensions)
        .build()


    private fun Coord.Builder.setTs(date: Date): Coord.Builder {
        ts = SimpleDateFormat(TS_PATTERN, Locale.ENGLISH).format(date)
        return this
    }

    private fun Coord.Builder.setOk(): Coord.Builder {
        failureReason = FailureReason.newBuilder()
            .setCode(FailureReasonCode.OK)
            .build()
        return this
    }

    private fun Coord.Builder.setLocation(location: org.findmykids.geo.data.model.Location): Coord.Builder {
        addLocations(
            Location.newBuilder()
                .setProvider(
                    when (location.source) {
                        LocationSource.GOOGLE_FUSED -> Provider.FUSED
                        LocationSource.HUAWEI_FUSED -> Provider.FUSED
                        LocationSource.LOCATION_MANAGER -> Provider.GPS
                    }
                )
                .setLatitude(location.latitude)
                .setLongitude(location.longitude)
                .setAge(System.currentTimeMillis() - (location.elapsedRealtimeNanos ?: location.time))
                .apply { location.accuracy?.let { accuracy = it.toLong() } }
                .apply { location.altitude?.let { altitude = it.toFloat() } }
                .apply { location.speed?.let { speed = it } }
                .apply { location.bearing?.let { cource = it.toInt() } }
                .build()
        )
        return this
    }

    private fun Coord.Builder.setBattery(battery: Battery?): Coord.Builder {
        if (battery == null) {
            batteryLevel = EMPTY_BATTERY_LEVEL_VALUE
        } else {
            isCharging = battery.isCharging
            batteryLevel = battery.level
        }
        return this
    }

    private fun Coord.Builder.setGps(gpsInfo: GpsInfo?): Coord.Builder {
        gpsInfo?.let {
            gps = Gps.newBuilder()
                .setTimeToFirstFix(gpsInfo.timeToFirstFix)
                .setTotalSatellitesCount(gpsInfo.maxSatellites)
                .setUsedSatellitesCount(gpsInfo.getUsedSatellitesCount())
                .setNotUsedSatellitesCount(gpsInfo.maxSatellites - gpsInfo.satellites.size)
                .build()
        }
        return this
    }

    private fun Coord.Builder.setWifi(wifi: org.findmykids.geo.data.model.Wifi?): Coord.Builder {
        wifi?.let {
            addWifi(
                Wifi.newBuilder()
                    .setMac(it.mac)
                    .setSignalStrength(it.level)
                    .setState(true)
                    .setName(it.ssid)
                    .build()
            )
        }
        return this
    }

    private fun Coord.Builder.setReason(session: Session): Coord.Builder {
        reason = when (session.commandsTypes.first()) {
            CommandType.ACTIVATE -> Reason.ACTIVATE
            CommandType.DEACTIVATE -> Reason.DEACTIVATE
            CommandType.PING -> Reason.PING
            CommandType.REQUEST -> Reason.REQUEST
            CommandType.TIMER -> Reason.TIMER
            CommandType.BOOT -> Reason.BOOT
            CommandType.ACTIVITY -> Reason.ACTIVITYTRANSITIONS
            CommandType.ZONE -> Reason.ZONE
            CommandType.STATION -> Reason.STATION
            CommandType.PASSIVE -> Reason.PASSIVE
            CommandType.REALTIME -> Reason.PING
            CommandType.INVALID -> Reason.LOG
        }
        return this
    }

    private fun Coord.Builder.setActivity(activity: org.findmykids.geo.data.model.Activity?): Coord.Builder {
        activityType = Activity.UNKNOWN
        activity?.let {
            if (it.activitiesWithConfidences.isNotEmpty()) {
                val (activityTypeValue, confidenceValue) = it.activitiesWithConfidences.maxBy { pair -> pair.value }!!
                activityType = when (activityTypeValue) {
                    ActivityType.IN_VEHICLE -> Activity.IN_VEHICLE
                    ActivityType.ON_BICYCLE -> Activity.ON_BICYCLE
                    ActivityType.ON_FOOT -> Activity.ON_FOOT
                    ActivityType.STILL -> Activity.STILL
                    ActivityType.UNKNOWN -> Activity.UNKNOWN
                    ActivityType.TILTING -> Activity.TILTING
                    ActivityType.WALKING -> Activity.WALKING
                    ActivityType.RUNNING -> Activity.RUNNING
                }
                confidence = confidenceValue
            }
        }
        return this
    }

    private fun Coord.Builder.setExtensions(sendGeoLocation: SendGeoLocation, geoExtensions: GeoExtensions): Coord.Builder {
        geoExtensions
            .mapExtensions(sendGeoLocation)
            .map {
                Extension
                    .newBuilder()
                    .setData(it.toByteString())
                    .setName(it.javaClass.`package`!!.name)
                    .build()
            }
            .forEach {
                addExtension(it)
            }
        return this
    }
}