package org.findmykids.geo.data.model

import androidx.annotation.IntDef
import com.google.android.gms.location.DetectedActivity
import com.google.android.gms.location.LocationRequest
import java.util.concurrent.TimeUnit


internal sealed class Configuration(
    val type: Int
) {
    data class ActivityConfiguration(
        /**
         * Какие типы активностей отслеживать
         */
        val activities: List<Int> = ACTIVITY_DEFAULT_ACTIVITIES
    ) : Configuration(Type.ACTIVITY.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$activities)"
        }
    }

    data class PassiveConfiguration(
        /**
         * Приоритет получения координат
         */
        @PassivePriority
        val priority: Int = LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY,
        /**
         * Минимальное время между получением координат
         */
        val minTime: Long = PASSIVE_DEFAULT_MIN_TIME,
        /**
         * Минимальная дистанция получения координат
         */
        val minDistance: Float = PASSIVE_DEFAULT_MIN_DISTANCE
    ) : Configuration(Type.PASSIVE.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[2].name}=$priority, " +
                    "${this::class.java.declaredFields[1].name}=$minTime, " +
                    "${this::class.java.declaredFields[0].name}=$minDistance)"
        }
    }

    data class StationConfiguration(
        /**
         * Меньший радиус области вокруг текущей координаты, на которой необходимо ловить событие выхода из неё
         */
        val lowRadius: Float = STATION_DEFAULT_LOW_RADIUS,
        /**
         * Больший радиус области вокруг текущей координаты, на которой необходимо ловить событие выхода из неё
         */
        val highRadius: Float = STATION_DEFAULT_HIGH_RADIUS
    ) : Configuration(Type.STATION.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[1].name}=$lowRadius, " +
                    "${this::class.java.declaredFields[0].name}=$highRadius)"
        }
    }

    data class TimerConfiguration(
        /**
         * Через какое время запустить сервис
         */
        val delay: Long = TIMER_DEFAULT_TIMER_DELAY
    ) : Configuration(Type.TIMER.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$delay)"
        }
    }


    data class LocationDataConfiguration(
        /**
         * Приоритет получения координат
         */
        @PassivePriority
        val priority: Int = LocationRequest.PRIORITY_HIGH_ACCURACY,
        /**
         * Интервал между получением координат
         */
        val interval: Long = LOCATION_DATA_DEFAULT_INTERVAL,
        /**
         * Максимальное время между получением координат
         */
        val maxWaitTime: Long = LOCATION_DATA_DEFAULT_MAX_WAIT_TIME,
        /**
         * Максимально быстрый интервал между получением координат
         */
        val fastestInterval: Long = LOCATION_DATA_DEFAULT_FASTEST_INTERVAL,
        /**
         * Минимальная дистанция получения координат
         */
        val smallestDisplacement: Float = LOCATION_DATA_DEFAULT_SMALLEST_DISPLACEMENT,
        /**
         * Через какое время перезапустить Fused менеджер, если произошла ошибка
         */
        val restartDelay: Long = LOCATION_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.LOCATION_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[3].name}=$priority, " +
                    "${this::class.java.declaredFields[1].name}=$interval, " +
                    "${this::class.java.declaredFields[2].name}=$maxWaitTime, " +
                    "${this::class.java.declaredFields[0].name}=$fastestInterval, " +
                    "${this::class.java.declaredFields[5].name}=$smallestDisplacement, " +
                    "${this::class.java.declaredFields[4].name}=$restartDelay)"
        }
    }

    data class GpsDataConfiguration(
        /**
         * Через какое время перезапустить GPS менеджер, если произошла ошибка
         */
        val restartDelay: Long = GPS_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.GPS_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$restartDelay)"
        }
    }

    data class BatteryDataConfiguration(
        /**
         * Через какое время перезапустить battery менеджер, если произошла ошибка
         */
        val restartDelay: Long = BATTERY_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.BATTERY_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$restartDelay)"
        }
    }

    data class LbsDataConfiguration(
        /**
         * Через какое время перезапустить LBS менеджер, если произошла ошибка
         */
        val restartDelay: Long = LBS_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.LBS_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$restartDelay)"
        }
    }

    data class RemoteDataConfiguration(
        /**
         * время ожидание первого переподключения
         */
        val startDelay: Long = REMOTE_DATA_DEFAULT_START_DELAY,
        /**
         * время ожидание последующих переподключений
         */
        val stepDelay: Long = REMOTE_DATA_DEFAULT_STEP_DELAY,
        /**
         * Максимальное время ожидания переподключения
         */
        val maxDelay: Long = REMOTE_DATA_DEFAULT_MAX_DELAY,
        /**
         * Время ожидания во время реалтайма
         */
        val realtimeDelay: Long = REMOTE_DATA_DEFAULT_REALTIME_DELAY
    ) : Configuration(Type.REMOTE_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[2].name}=$startDelay, " +
                    "${this::class.java.declaredFields[3].name}=$stepDelay, " +
                    "${this::class.java.declaredFields[0].name}=$maxDelay, " +
                    "${this::class.java.declaredFields[1].name}=$realtimeDelay)"
        }
    }

    data class SensorsDataConfiguration(
        /**
         * С каким периодом получать данные от сенсоров (Events may be received faster or slower than the specified rate.)
         */
        val samplingPeriodUs: Int = SENSOR_DATA_DEFAULT_SAMPLING_PERIOD_US
    ) : Configuration(Type.SENSORS_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$samplingPeriodUs)"
        }
    }

    data class TimeoutDataConfiguration(
        /**
         * Через какое время сообщить что время работы истекло и пора закрывать сервис
         */
        val delay: Long = TIMEOUT_DATA_DEFAULT_TIMEOUT_DELAY,
        /**
         * Через какое время перезапустить таймер, если произошла ошибка
         */
        val restartDelay: Long = TIMEOUT_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.TIMEOUT_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$delay, " +
                    "${this::class.java.declaredFields[1].name}=$restartDelay)"
        }
    }

    data class WifiDataConfiguration(
        /**
         * Через какое время перезапустить WiFi менеджер, если произошла ошибка
         */
        val restartDelay: Long = WIFI_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.WIFI_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$restartDelay)"
        }
    }

    data class ActivityDataConfiguration(
        /**
         * Через какое время получить данные об активности
         */
        val detectionIntervalMillis: Long = ACTIVITY_DATA_DEFAULT_ACTIVITY_DETECTION_INTERVAL_MILLIS,
        /**
         * Через какое время перезапустить Activity recognition client, если произошла ошибка
         */
        val restartDelay: Long = ACTIVITY_DATA_DEFAULT_RESTART_DELAY
    ) : Configuration(Type.ACTIVITY_DATA.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$restartDelay)"
        }
    }


    data class GeoStorageConfiguration(
        /**
         * По сколько отправлять офлайн координаты
         */
        val offlineMaxCount: Int = GEO_STORAGE_DEFAULT_OFFLINE_MAX_COUNT,
        /**
         * Время ожидание перед отправкой следующей пачки оффлайновых координта
         */
        val timeoutOnSendOffline: Long = GEO_STORAGE_DEFAULT_TIMEOUT_ON_SEND_OFFLINE
    ) : Configuration(Type.GEO_STORAGE.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$offlineMaxCount, " +
                    "${this::class.java.declaredFields[1].name}=$timeoutOnSendOffline)"
        }
    }

    data class ErrorStorageConfiguration(
        /**
         * По сколько отправлять офлайн ошибок
         */
        val offlineMaxCount: Int = ERROR_STORAGE_DEFAULT_OFFLINE_MAX_COUNT,
        /**
         * Время ожидание перед отправкой следующей пачки оффлайновых ошибок
         */
        val timeoutOnSendOffline: Long = ERROR_STORAGE_DEFAULT_TIMEOUT_ON_SEND_OFFLINE
    ) : Configuration(Type.ERROR_STORAGE.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[0].name}=$offlineMaxCount, " +
                    "${this::class.java.declaredFields[1].name}=$timeoutOnSendOffline)"
        }
    }


    data class LocationLiveConfiguration(
        /**
         * Критерий не остановки работы сервиса из-за малого количества Fused геоданных
         * Если Fused геоданных меньше geoCountDontStopCriteria, то не останавливаем сервис
         */
        val geoCountDontStopCriteria: Int = LOCATION_LIVE_DEFAULT_GEO_COUNT_DONT_STOP_CRITERIA,
        /**
         * Фильтруем по времени Fused геоданные, для приминения критериев остановки distanceStopCriteria
         * Сделано для того чтобы не учитывать все Fused геоданные, а лишь их часть (отправленные и не отправленные)
         */
        val filterGeoCountTime: Long = LOCATION_LIVE_DEFAULT_FILTER_GEO_COUNT,
        /**
         * Критерий остановки работы сервиса из-за незначитального изменения дистанции за время filterGeoCountTime
         * Если дистанция изменилась меньше чем distanceStopCriteria, то останавливаем сервис
         */
        val distanceStopCriteria: Int = LOCATION_LIVE_DEFAULT_DISTANCE_STOP_CRITERIA,
        /**
         * Критерий НЕ остановки работы сервиса из-за большой точности гео
         * Если точность больше чем accuracyDontStopCriteria, то НЕ останавливаем сервис
         */
        val accuracyDontStopCriteria: Int = LOCATION_LIVE_DEFAULT_ACCURACY_DONT_STOP_CRITERIA,
        /**
         * Критерий отправки Fused геоданных из-за количества накопившихся не отправленных Fused геоданных
         * Если не отправленных Fused геоданных стало больше или равно countNotSendedGeoSendCriteria, то отправляем
         */
        val countNotSendedGeoSendCriteria: Int = LOCATION_LIVE_DEFAULT_COUNT_NOT_SENDED_GEO_SEND_CRITERIA,
        /**
         * Критерий отправки Fused геоданных из-за изменения дистанции
         * если дистанция между новой и старой < distanceSendCriteria, то отправляем
         */
        val distanceSendCriteria: Int = LOCATION_LIVE_DEFAULT_DISTANCE_SEND_CRITERIA,
        /**
         * Критерий отправки Fused геоданных из-за изменения точности
         * если новая/старая < accuracySendCriteria, то отправляем
         */
        val accuracySendCriteria: Float = LOCATION_LIVE_DEFAULT_ACCURACY_SEND_CRITERIA,
        /**
         * Критерий отправки Fused геоданных из-за изменения направления движения
         * если разница между новым и старым неаправлением > bearingChangeSendCriteria, то отправляем
         */
        val bearingChangeSendCriteria: Float = LOCATION_LIVE_DEFAULT_BEARING_CHANGE_SEND_CRITERIA,
        /**
         * Ограничение для использования критерия bearingChangeSendCriteria по скорости,
         * Чтобы исключить вращение телефона на месте
         */
        val bearingChangeSpeedSendCriteria: Float = LOCATION_LIVE_DEFAULT_BEARING_CHANGE_SPEED_SEND_CRITERIA
    ) : Configuration(Type.LOCATION_LIVE.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[8].name}=$geoCountDontStopCriteria, " +
                    "${this::class.java.declaredFields[7].name}=$filterGeoCountTime, " +
                    "${this::class.java.declaredFields[6].name}=$distanceStopCriteria, " +
                    "${this::class.java.declaredFields[0].name}=$accuracyDontStopCriteria, " +
                    "${this::class.java.declaredFields[4].name}=$countNotSendedGeoSendCriteria, " +
                    "${this::class.java.declaredFields[5].name}=$distanceSendCriteria, " +
                    "${this::class.java.declaredFields[1].name}=$accuracySendCriteria, " +
                    "${this::class.java.declaredFields[2].name}=$bearingChangeSendCriteria, " +
                    "${this::class.java.declaredFields[3].name}=$bearingChangeSpeedSendCriteria)"
        }
    }

    data class LocatorLiveConfiguration(
        /**
         * Ктритерий игнорирования координаты если ее точность больше указанной
         */
        val accuracyIgnoreCriteria: Int = LOCATOR_LIVE_DEFAULT_ACCURACY_IGNORE_CRITERIA,
        /**
         * Критерий отправки геоданных из-за количества накопившихся не отправленных геоданных
         * Если не отправленных Gps геоданных стало больше или равно countNotSendedGeoSendCriteria, то отправляем
         */
        val countNotSendedGeoSendCriteria: Int = LOCATOR_LIVE_DEFAULT_COUNT_NOT_SENDED_GEO_SEND_CRITERIA,
        /**
         * Критерий отправки геоданных из-за изменения дистанции
         * если дистанция между новой и старой < distanceSendCriteria, то отправляем
         */
        val distanceSendCriteria: Int = LOCATOR_LIVE_DEFAULT_DISTANCE_SEND_CRITERIA,
        /**
         * Критерий отправки геоданных из-за изменения точности
         * если новая/старая < accuracySendCriteria, то отправляем
         */
        val accuracySendCriteria: Float = LOCATOR_LIVE_DEFAULT_ACCURACY_SEND_CRITERIA
    ) : Configuration(Type.LOCATOR_LIVE.value) {
        override fun toString(): String {
            return "${this::class.simpleName}(" +
                    "${this::class.java.declaredFields[1].name}=$countNotSendedGeoSendCriteria, " +
                    "${this::class.java.declaredFields[2].name}=$distanceSendCriteria, " +
                    "${this::class.java.declaredFields[0].name}=$accuracySendCriteria)"
        }
    }


    override fun toString(): String = ""


    enum class Type(val value: Int) {
        ACTIVITY(1),
        PASSIVE(2),
        STATION(3),
        TIMER(4),
        LOCATION_DATA(5),
        GPS_DATA(6),
        LBS_DATA(7),
        REMOTE_DATA(8),
        SENSORS_DATA(9),
        TIMEOUT_DATA(10),
        WIFI_DATA(11),
        GEO_STORAGE(12),
        LOCATION_LIVE(13),
        LOCATOR_LIVE(14),
        ACTIVITY_DATA(15),
        BATTERY_DATA(16),
        ERROR_STORAGE(17),
    }


    companion object {
        @IntDef(
            LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY,
            LocationRequest.PRIORITY_HIGH_ACCURACY,
            LocationRequest.PRIORITY_LOW_POWER,
            LocationRequest.PRIORITY_NO_POWER
        )
        @Retention(AnnotationRetention.SOURCE)
        annotation class PassivePriority

        private val ACTIVITY_DEFAULT_ACTIVITIES = listOf(
            DetectedActivity.STILL,
            DetectedActivity.ON_FOOT,
            DetectedActivity.WALKING,
            DetectedActivity.IN_VEHICLE,
            DetectedActivity.ON_BICYCLE,
            DetectedActivity.RUNNING
        )

        private var PASSIVE_DEFAULT_MIN_TIME = TimeUnit.MINUTES.toMillis(10)
        private const val PASSIVE_DEFAULT_MIN_DISTANCE = 30F

        private const val STATION_DEFAULT_LOW_RADIUS = 30f
        private const val STATION_DEFAULT_HIGH_RADIUS = 150f

        private var TIMER_DEFAULT_TIMER_DELAY = TimeUnit.MINUTES.toMillis(15)


        private var LOCATION_DATA_DEFAULT_INTERVAL = TimeUnit.SECONDS.toMillis(5)
        private var LOCATION_DATA_DEFAULT_MAX_WAIT_TIME = TimeUnit.SECONDS.toMillis(1)
        private var LOCATION_DATA_DEFAULT_FASTEST_INTERVAL = LOCATION_DATA_DEFAULT_INTERVAL / 2
        private const val LOCATION_DATA_DEFAULT_SMALLEST_DISPLACEMENT = 1.0f
        private var LOCATION_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(2)

        private var GPS_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(10)

        private var BATTERY_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(10)

        private var LBS_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(10)

        private var REMOTE_DATA_DEFAULT_START_DELAY = TimeUnit.SECONDS.toMillis(60)
        private var REMOTE_DATA_DEFAULT_STEP_DELAY = TimeUnit.SECONDS.toMillis(30)
        private var REMOTE_DATA_DEFAULT_MAX_DELAY = TimeUnit.MINUTES.toMillis(5)
        private var REMOTE_DATA_DEFAULT_REALTIME_DELAY = TimeUnit.SECONDS.toMillis(1)

        private const val SENSOR_DATA_DEFAULT_SENSOR_FREQ_HZ = 10.0
        private const val SENSOR_DATA_DEFAULT_SAMPLING_PERIOD_US = (1.0e6 / SENSOR_DATA_DEFAULT_SENSOR_FREQ_HZ).toInt()

        private var TIMEOUT_DATA_DEFAULT_TIMEOUT_DELAY = TimeUnit.SECONDS.toMillis(45)
        private var TIMEOUT_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(15)

        private var WIFI_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(30)

        private var ACTIVITY_DATA_DEFAULT_ACTIVITY_DETECTION_INTERVAL_MILLIS = TimeUnit.SECONDS.toMillis(5)
        private var ACTIVITY_DATA_DEFAULT_RESTART_DELAY = TimeUnit.SECONDS.toMillis(30)


        private const val GEO_STORAGE_DEFAULT_OFFLINE_MAX_COUNT = 100
        private var GEO_STORAGE_DEFAULT_TIMEOUT_ON_SEND_OFFLINE = TimeUnit.SECONDS.toMillis(10)

        private const val ERROR_STORAGE_DEFAULT_OFFLINE_MAX_COUNT = 100
        private var ERROR_STORAGE_DEFAULT_TIMEOUT_ON_SEND_OFFLINE = TimeUnit.SECONDS.toMillis(10)


        private const val LOCATION_LIVE_DEFAULT_GEO_COUNT_DONT_STOP_CRITERIA = 3
        private var LOCATION_LIVE_DEFAULT_FILTER_GEO_COUNT = TimeUnit.SECONDS.toMillis(60)
        private const val LOCATION_LIVE_DEFAULT_DISTANCE_STOP_CRITERIA = 5
        private const val LOCATION_LIVE_DEFAULT_ACCURACY_DONT_STOP_CRITERIA = 100
        private const val LOCATION_LIVE_DEFAULT_COUNT_NOT_SENDED_GEO_SEND_CRITERIA = 10
        private const val LOCATION_LIVE_DEFAULT_DISTANCE_SEND_CRITERIA = 10
        private const val LOCATION_LIVE_DEFAULT_ACCURACY_SEND_CRITERIA = 0.5f
        private const val LOCATION_LIVE_DEFAULT_BEARING_CHANGE_SEND_CRITERIA = 70f
        private const val LOCATION_LIVE_DEFAULT_BEARING_CHANGE_SPEED_SEND_CRITERIA = 0.5f

        private const val LOCATOR_LIVE_DEFAULT_ACCURACY_IGNORE_CRITERIA = 200
        private const val LOCATOR_LIVE_DEFAULT_COUNT_NOT_SENDED_GEO_SEND_CRITERIA = 10
        private const val LOCATOR_LIVE_DEFAULT_DISTANCE_SEND_CRITERIA = 10
        private const val LOCATOR_LIVE_DEFAULT_ACCURACY_SEND_CRITERIA = 0.5f
    }
}