package org.findmykids.geo.presentation.session.output

import io.reactivex.Single
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.GeoLocation
import org.findmykids.geo.domain.authorization.AuthorizationInteractor
import org.findmykids.geo.domain.model.InnerEvent
import org.findmykids.geo.domain.subscriber.activity.ActivitySubscriberInteractor
import org.findmykids.geo.domain.subscriber.passive.PassiveSubscriberInteractor
import org.findmykids.geo.domain.subscriber.station.StationSubscriberInteractor
import org.findmykids.geo.domain.subscriber.timer.TimerSubscriberInteractor
import org.findmykids.geo.domain.subscriber.zones.ZonesSubscriberInteractor
import javax.inject.Inject


internal class EventFrontControllerImpl @Inject constructor(
    private val mAuthorizationInteractor: AuthorizationInteractor,
    private val mTimerSubscriberInteractor: TimerSubscriberInteractor,
    private val mPassiveSubscriberInteractor: PassiveSubscriberInteractor,
    private val mZonesSubscriberInteractor: ZonesSubscriberInteractor,
    private val mActivitySubscriberInteractor: ActivitySubscriberInteractor,
    private val mStationSubscriberInteractor: StationSubscriberInteractor
) : EventFrontController {

    override fun toString(): String = ""


    override fun isNeedStopService(innerEvent: InnerEvent): Single<Boolean> {
        Logger.i().addArg(innerEvent).print()
        return when (innerEvent) {
            is InnerEvent.NewGeo -> processNewGeo()
            is InnerEvent.NewError -> processNewError()
            is InnerEvent.NewGeoLocation -> processNewGeoLocation(innerEvent.geoLocation)
            is InnerEvent.NewZones -> processNewZones()
            is InnerEvent.NewConfigurations -> processNewConfiguration()
            is InnerEvent.EndCurrentSession -> processEndCurrentSession()
            InnerEvent.Stop -> processStop()
        }.doOnSuccess {
            Logger.d().with(this@EventFrontControllerImpl).setResult(it).print()
        }
    }


    private fun processNewGeo(): Single<Boolean> = Single.just(false)

    private fun processNewError(): Single<Boolean> = Single.just(false)

    private fun processNewGeoLocation(geoLocation: GeoLocation): Single<Boolean> = mStationSubscriberInteractor
        .subscribe(geoLocation.location)
        .toSingleDefault(false)

    private fun processNewZones(): Single<Boolean> = mZonesSubscriberInteractor
        .subscribe()
        .toSingleDefault(false)

    private fun processNewConfiguration(): Single<Boolean> = mTimerSubscriberInteractor
        .subscribe()
        .andThen(mPassiveSubscriberInteractor.subscribe())
        .andThen(mActivitySubscriberInteractor.subscribe())
        .toSingleDefault(false)

    private fun processEndCurrentSession(): Single<Boolean> = mAuthorizationInteractor
        .isActivated()
        .flatMap { isActivated ->
            if (isActivated) {
                mTimerSubscriberInteractor
                    .subscribeIfNeed()
                    .andThen(mPassiveSubscriberInteractor.subscribeIfNeed())
                    .andThen(mActivitySubscriberInteractor.subscribeIfNeed())
                    .andThen(mZonesSubscriberInteractor.subscribeIfNeed())
                    .toSingleDefault(true)
            } else {
                Single.just(true)
            }
        }

    private fun processStop(): Single<Boolean> = mTimerSubscriberInteractor
        .unsubscribe()
        .andThen(mPassiveSubscriberInteractor.unsubscribe())
        .andThen(mActivitySubscriberInteractor.unsubscribe())
        .andThen(mZonesSubscriberInteractor.unsubscribe())
        .andThen(mStationSubscriberInteractor.unsubscribe())
        .toSingleDefault(true)
}