package org.findmykids.geo.domain.live.remote

import io.reactivex.Completable
import io.reactivex.Observable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configurations
import org.findmykids.geo.data.model.Zones
import org.findmykids.geo.data.repository.live.activity.ActivityRepository
import org.findmykids.geo.data.repository.live.battery.BatteryRepository
import org.findmykids.geo.data.repository.live.gps.GpsRepository
import org.findmykids.geo.data.repository.live.lbs.LbsRepository
import org.findmykids.geo.data.repository.live.location.LocationRepository
import org.findmykids.geo.data.repository.live.remote.RemoteEvent
import org.findmykids.geo.data.repository.live.remote.RemoteRepository
import org.findmykids.geo.data.repository.live.sensors.SensorsRepository
import org.findmykids.geo.data.repository.live.timeout.TimeoutRepository
import org.findmykids.geo.data.repository.live.wifi.WifiRepository
import org.findmykids.geo.data.repository.storage.authorisation.AuthorisationRepository
import org.findmykids.geo.data.repository.storage.configuration.ConfigurationRepository
import org.findmykids.geo.data.repository.storage.currentSession.CurrentSessionRepository
import org.findmykids.geo.data.repository.storage.error.ErrorRepository
import org.findmykids.geo.data.repository.storage.geo.GeoRepository
import org.findmykids.geo.data.repository.storage.request.RequestRepository
import org.findmykids.geo.data.repository.storage.zones.ZonesRepository
import org.findmykids.geo.domain.BaseInteractor
import org.findmykids.geo.domain.model.InnerEvent
import javax.inject.Inject


internal class RemoteInteractorImpl @Inject constructor(
    private val mRemoteRepository: RemoteRepository,
    private val mGeoRepository: GeoRepository,
    private val mErrorRepository: ErrorRepository,
    private val mTimeoutRepository: TimeoutRepository,
    private val mAuthorisationRepository: AuthorisationRepository,
    private val mZoneRepository: ZonesRepository,
    private val mConfigurationRepository: ConfigurationRepository,
    private val mCurrentSessionRepository: CurrentSessionRepository,
    private val mRequestRepository: RequestRepository,
    private val mLocationRepository: LocationRepository,
    private val mBatteryRepository: BatteryRepository,
    private val mActivityRepository: ActivityRepository,
    private val mGpsRepository: GpsRepository,
    private val mSensorsRepository: SensorsRepository,
    private val mLbsRepository: LbsRepository,
    private val mWifiRepository: WifiRepository
) : BaseInteractor(), RemoteInteractor {

    override fun toString(): String = ""


    override fun update(): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .flatMapObservable { mRemoteRepository.observeEvents(it.socketDataConfiguration) }
        .concatMap {
            Logger.d().addArg(it).with(this@RemoteInteractorImpl).print()
            when (it) {
                is RemoteEvent.Created -> created()
                is RemoteEvent.Deactivate -> deactivate()
                is RemoteEvent.Request -> request()
                is RemoteEvent.ReRequest -> reRequest()
                is RemoteEvent.ConfigurationsChanged -> configurationChanged(it.configurations)
                is RemoteEvent.ZoneChanged -> zoneChanged(it.zones)
                is RemoteEvent.RealtimeBegin -> realtimeBegin()
                is RemoteEvent.RealtimeEnd -> realtimeEnd()
            }
        }


    private fun created(): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .flatMap { configurations ->
            mCurrentSessionRepository
                .getSession()
                .map { Pair(configurations, it) }
        }
        .flatMapCompletable { (configurations, session) ->
            mGeoRepository
                .sendAll(session, configurations.geoStorageConfiguration)
                .andThen(mErrorRepository.sendAll(session, configurations.errorStorageConfiguration))
                .subscribeOn(Schedulers.newThread())
        }
        .toObservable()

    private fun deactivate(): Observable<InnerEvent> = mAuthorisationRepository
        .clear()
        .andThen { mZoneRepository.clear() }
        .andThen { mConfigurationRepository.clear() }
        .andThen { mGeoRepository.clear() }
        .andThen(Observable.just(InnerEvent.Stop as InnerEvent))

    private fun request(): Observable<InnerEvent> = mGeoRepository
        .getLastSaveGeoLocation()
        .flatMapCompletable {
            when (it.value) {
                null -> mRequestRepository.sendNoGeo()
                else -> mGeoRepository.sendGeoLocation(it.value, "Request")
            }
        }
        .toObservable()

    private fun reRequest(): Observable<InnerEvent> = mGeoRepository
        .getLastSaveGeoLocation()
        .flatMap { lastSaveGeoLocation ->
            mGeoRepository
                .getLastSendGeoLocation()
                .map { Pair(lastSaveGeoLocation, it) }
        }
        .flatMapCompletable { (lastSaveGeoLocation, lastSendGeoLocation) ->
            Logger.d().addArg(lastSaveGeoLocation).addArg(lastSendGeoLocation).with(this@RemoteInteractorImpl).print()
            when (lastSaveGeoLocation.value) {
                null -> mRequestRepository.sendNoGeo()
                lastSendGeoLocation.value -> mRequestRepository.sendGeoNotChanged()
                else -> mGeoRepository.sendGeoLocation(lastSaveGeoLocation.value, "ReRequest")
            }
        }
        .toObservable()

    private fun configurationChanged(newConfigurations: Configurations): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .flatMapObservable { oldConfiguration ->
            Logger.d().addArg(newConfigurations).addArg(oldConfiguration).with(this@RemoteInteractorImpl).print()
            if (newConfigurations == oldConfiguration) {
                Logger.d("configurations are equals").with(this@RemoteInteractorImpl).print()
                Observable.empty()
            } else {
                mConfigurationRepository
                    .set(newConfigurations)
                    .andThen(
                        if (newConfigurations.activityDataConfiguration == oldConfiguration.activityDataConfiguration) {
                            Completable.complete()
                        } else {
                            mActivityRepository.restart(newConfigurations.activityDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.batteryDataConfiguration == oldConfiguration.batteryDataConfiguration) {
                            Completable.complete()
                        } else {
                            mBatteryRepository.restart(newConfigurations.batteryDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.gpsDataConfiguration == oldConfiguration.gpsDataConfiguration) {
                            Completable.complete()
                        } else {
                            mGpsRepository.restart(newConfigurations.gpsDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.lbsDataConfiguration == oldConfiguration.lbsDataConfiguration) {
                            Completable.complete()
                        } else {
                            mLbsRepository.restart(newConfigurations.lbsDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.locationDataConfiguration == oldConfiguration.locationDataConfiguration) {
                            Completable.complete()
                        } else {
                            mLocationRepository.restart(newConfigurations.locationDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.sensorsDataConfiguration == oldConfiguration.sensorsDataConfiguration) {
                            Completable.complete()
                        } else {
                            mSensorsRepository.restart(newConfigurations.sensorsDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.timeoutDataConfiguration == oldConfiguration.timeoutDataConfiguration) {
                            Completable.complete()
                        } else {
                            mTimeoutRepository.restart(newConfigurations.timeoutDataConfiguration)
                        }
                    )
                    .andThen(
                        if (newConfigurations.wifiDataConfiguration == oldConfiguration.wifiDataConfiguration) {
                            Completable.complete()
                        } else {
                            mWifiRepository.restart(newConfigurations.wifiDataConfiguration)
                        }
                    )
                    .toObservable<InnerEvent>()
                    .map { InnerEvent.NewConfigurations(newConfigurations) }
            }
        }

    private fun zoneChanged(newZones: Zones): Observable<InnerEvent> = mZoneRepository
        .get()
        .flatMapObservable { oldZones ->
            if (newZones == oldZones) {
                Observable.empty()
            } else {
                mZoneRepository
                    .set(newZones)
                    .andThen(Observable.just(InnerEvent.NewZones(newZones)))
            }
        }

    private fun realtimeBegin(): Observable<InnerEvent> = mCurrentSessionRepository
        .setRealtime(true)
        .toObservable()

    private fun realtimeEnd(): Observable<InnerEvent> = mCurrentSessionRepository
        .setRealtime(false)
        .toObservable()
}