package org.findmykids.geo.domain.live.location

import io.reactivex.Completable
import io.reactivex.Observable
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.MathUtil
import org.findmykids.geo.data.model.*
import org.findmykids.geo.data.repository.live.activity.ActivityRepository
import org.findmykids.geo.data.repository.live.battery.BatteryRepository
import org.findmykids.geo.data.repository.live.gps.GpsRepository
import org.findmykids.geo.data.repository.live.location.LocationRepository
import org.findmykids.geo.data.repository.live.wifi.WifiRepository
import org.findmykids.geo.data.repository.storage.configuration.ConfigurationRepository
import org.findmykids.geo.data.repository.storage.currentSession.CurrentSessionRepository
import org.findmykids.geo.data.repository.storage.geo.GeoRepository
import org.findmykids.geo.domain.BaseInteractor
import org.findmykids.geo.domain.model.InnerEvent
import java.util.*
import javax.inject.Inject


internal class LocationInteractorImpl @Inject constructor(
    private val mGeoRepository: GeoRepository,
    private val mCurrentSessionRepository: CurrentSessionRepository,
    private val mConfigurationRepository: ConfigurationRepository,
    private val mActivityRepository: ActivityRepository,
    private val mBatteryRepository: BatteryRepository,
    private val mGpsRepository: GpsRepository,
    private val mLocationRepository: LocationRepository,
    private val mWifiRepository: WifiRepository
) : BaseInteractor(), LocationInteractor {

    private var mActivity: Activity? = null
    private var mGpsInfo: GpsInfo? = null
    private var mBattery: Battery? = null

    /**
     * Сохраняем все геолокации
     */
    private val mAllGeoLocations = LinkedList<GeoLocation>()

    /**
     * Последняя оправленныя геолокация
     */
    private var mLastSendedGeoLocation: GeoLocation? = null

    /**
     * Список не отправленнных геолокация, чистится при отправке геолокации
     */
    private var mNotSended = mutableListOf<GeoLocation>()


    override fun toString(): String = ""

    override fun update(): Observable<InnerEvent> = mConfigurationRepository
        .get()
        .doOnSubscribe {
            Logger.d().with(this@LocationInteractorImpl).print()
            mActivity = null
            mGpsInfo = null
            mBattery = null
            mNotSended.clear()
            mLastSendedGeoLocation = null
        }
        .flatMapObservable {
            // подписка, по факту только locationEmitter что либо пришлет
            Logger.d().addArg(it).with(this@LocationInteractorImpl).print()
            Observable.mergeArray(
                activityEmitter(it.activityDataConfiguration),
                batteryEmitter(it.batteryDataConfiguration),
                gpsEmitter(it.gpsDataConfiguration),
                locationEmitter(it.locationDataConfiguration)
            )
        }
        .concatMap { geoLocation ->
            // получаем актуальную конфигурацию
            Logger.d().addArg(geoLocation).with(this@LocationInteractorImpl).print()
            mConfigurationRepository
                .get()
                .flatMapObservable { configurations ->
                    // в зависимоти от конфигурации решаем что делать с геолокацией: отправляеть или нет
                    Logger.d().addArg(geoLocation).addArg(configurations).with(this@LocationInteractorImpl).print()
                    mAllGeoLocations.add(0, geoLocation)
                    when {
                        geoLocation.session.isRealtime -> sendGeoLocation(geoLocation, "Realtime")
                            .andThen(Observable.just(InnerEvent.NewGeoLocation(geoLocation)))
                        isStopCriteriaPass(geoLocation, configurations.locationLiveConfiguration) -> sendGeoLocation(geoLocation, "Last")
                            .andThen(
                                Observable.just(
                                    InnerEvent.NewGeoLocation(geoLocation),
                                    InnerEvent.EndCurrentSession(InnerEvent.EndCurrentSession.EndSessionReason.Fused)
                                )
                            )
                        else -> {
                            val reason = isSendCriteriaPass(geoLocation, configurations.locationLiveConfiguration)
                            if (reason != null) {
                                sendGeoLocation(geoLocation, reason).andThen(Observable.just(InnerEvent.NewGeoLocation(geoLocation)))
                            } else {
                                Logger.d("not send").with(this@LocationInteractorImpl).print()
                                mNotSended.add(geoLocation)
                                Observable.just(InnerEvent.NewGeoLocation(geoLocation))
                            }
                        }
                    }
                }
        }


    private fun activityEmitter(configuration: Configuration.ActivityDataConfiguration): Observable<GeoLocation> = mActivityRepository
        .observeEvents(configuration)
        .flatMap { event ->
            Logger.d().addArg(event).with(this@LocationInteractorImpl).print()
            mActivity = event.activity
            Observable.empty<GeoLocation>()
        }

    private fun batteryEmitter(configuration: Configuration.BatteryDataConfiguration): Observable<GeoLocation> = mBatteryRepository
        .observeEvents(configuration)
        .flatMap { event ->
            Logger.d().addArg(event).with(this@LocationInteractorImpl).print()
            mBattery = event.battery
            Observable.empty<GeoLocation>()
        }

    private fun gpsEmitter(configuration: Configuration.GpsDataConfiguration): Observable<GeoLocation> = mGpsRepository
        .observeEvents(configuration)
        .flatMap { event ->
            Logger.d().addArg(event).with(this@LocationInteractorImpl).print()
            mGpsInfo = event.gpsInfo
            Observable.empty<GeoLocation>()
        }

    private fun locationEmitter(configuration: Configuration.LocationDataConfiguration): Observable<GeoLocation> = mLocationRepository
        .observeEvents(configuration)
        .flatMapSingle { event ->
            Logger.d().addArg(event).with(this@LocationInteractorImpl).print()
            mCurrentSessionRepository
                .getSession()
                .flatMap { session ->
                    mWifiRepository
                        .getCurrentWifi()
                        .map { Pair(session, it.value) }
                }
                .map { (session, wifi) ->
                    GeoLocation(
                        Date(),
                        session,
                        event.location,
                        event.defineSessionIndex,
                        event.defineGlobalIndex,
                        null,
                        mGpsInfo,
                        mActivity,
                        mBattery,
                        wifi,
                        hashMapOf()
                    )
                }
        }


    private fun isStopCriteriaPass(geoLocation: GeoLocation, configuration: Configuration.LocationLiveConfiguration): Boolean {
        Logger.d().addArg(geoLocation).addArg(configuration).print()
        return when {
            geoLocation.location.accuracy != null && geoLocation.location.accuracy > configuration.accuracyDontStopCriteria -> false
            mAllGeoLocations.size < configuration.geoCountDontStopCriteria -> false
            else -> {
                var distance = 0f
                var previousLocation: Location? = null
                mAllGeoLocations
                    .filter { (System.currentTimeMillis() - it.create.time) <= configuration.filterGeoCountTime }
                    .forEach { geoLocationIt ->
                        previousLocation?.let {
                            distance += Location.distanceBetween(geoLocationIt.location, it)
                        }
                        previousLocation = geoLocationIt.location
                    }
                when {
                    distance < configuration.distanceStopCriteria -> {
                        Logger.d("Less distance").print()
                        true
                    }
                    else -> false
                }
            }
        }
    }


    private fun isSendCriteriaPass(geoLocation: GeoLocation, configuration: Configuration.LocationLiveConfiguration): String? {
        Logger.d().addArg(geoLocation).addArg(configuration).print()
        val lastSendedGeoLocation = mLastSendedGeoLocation
        val currentLocation = geoLocation.location
        // check first geo
        if (lastSendedGeoLocation == null) {
            return "First geo"
        }
        // check many not sended
        if (mNotSended.size > configuration.countNotSendedGeoSendCriteria) {
            return "Many not sended ${mNotSended.size}"
        }
        // check accuracy changed
        if (currentLocation.accuracy != null && lastSendedGeoLocation.location.accuracy != null) {
            val accuracy = currentLocation.accuracy / lastSendedGeoLocation.location.accuracy
            if (accuracy < configuration.accuracySendCriteria) {
                return "Accuracy changed to $accuracy"
            }
        }
        // check course changed
        if (lastSendedGeoLocation.location.bearing != null && currentLocation.bearing != null && currentLocation.speed != null &&
            currentLocation.speed > configuration.bearingChangeSpeedSendCriteria
        ) {
            val bearing = MathUtil.getDifference(lastSendedGeoLocation.location.bearing, currentLocation.bearing)
            if (bearing > configuration.bearingChangeSendCriteria) {
                return "Course changed to $bearing"
            }
        }
        // check distance changed
        val distance = Location.distanceBetween(lastSendedGeoLocation.location, currentLocation)
        if (distance > configuration.distanceSendCriteria) {
            return "Distance changed to $distance"
        }
        // check distance changed from not sended
        val notSended = mNotSended.firstOrNull { notSended ->
            Location.distanceBetween(notSended.location, currentLocation) > configuration.distanceSendCriteria
        }
        if (notSended != null) {
            return "Distance changed from not sended location to ${Location.distanceBetween(notSended.location, currentLocation)}"
        }
        // not send this location
        return null
    }


    private fun sendGeoLocation(geoLocation: GeoLocation, reason: String): Completable {
        Logger.d().addArg(reason).addArg(geoLocation).print()
        mNotSended.clear()
        mLastSendedGeoLocation = geoLocation
        return mGeoRepository.sendGeoLocation(geoLocation, reason)
    }
}