package org.findmykids.geo.data.repository.trigger.station.huawei

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import com.huawei.hmf.tasks.Tasks
import com.huawei.hms.location.Geofence
import com.huawei.hms.location.GeofenceRequest
import com.huawei.hms.location.LocationServices
import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Location
import org.findmykids.geo.data.repository.trigger.station.StationManager
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class StationHuaweiManagerImpl @Inject constructor(
    private val mContext: Context
) : StationManager {

    override fun toString(): String = ""


    override fun subscribe(location: Location, configuration: Configuration.StationConfiguration): Completable = Single
        .fromCallable {
            Logger.d().with(this@StationHuaweiManagerImpl).print()
            getIntent(PendingIntent.FLAG_UPDATE_CURRENT)?.let { intent ->
                Logger.d("has intent").with(this@StationHuaweiManagerImpl).print()
                val geofencingRequest = GeofenceRequest.Builder()
                    .setInitConversions(GeofenceRequest.EXIT_INIT_CONVERSION)
                    .createGeofence(
                        Geofence.Builder()
                            .setConversions(Geofence.EXIT_GEOFENCE_CONVERSION)
                            .setRoundArea(location.latitude, location.longitude, configuration.lowRadius)
                            .setUniqueId(GEOFENCE_STATION_ID_LOW)
                            .setValidContinueTime(Geofence.GEOFENCE_NEVER_EXPIRE)
                            .build()
                    )
                    .createGeofence(
                        Geofence.Builder()
                            .setConversions(Geofence.EXIT_GEOFENCE_CONVERSION)
                            .setRoundArea(location.latitude, location.longitude, configuration.highRadius)
                            .setUniqueId(GEOFENCE_STATION_ID_HIGH)
                            .setValidContinueTime(Geofence.GEOFENCE_NEVER_EXPIRE)
                            .build()
                    )
                    .build()
                try {
                    Tasks.await(LocationServices.getGeofenceService(mContext).createGeofenceList(geofencingRequest, intent), 10, TimeUnit.SECONDS)
                    true
                } catch (e: Exception) {
                    Logger.e(GeoException.StationSubscription(e)).with(this@StationHuaweiManagerImpl).print()
                    false
                }
            } ?: false
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@StationHuaweiManagerImpl).print()
        }
        .ignoreElement()

    override fun unsubscribe(): Completable = Single
        .fromCallable {
            Logger.d().with(this@StationHuaweiManagerImpl).print()
            getIntent(PendingIntent.FLAG_NO_CREATE)?.let { intent ->
                Logger.d("has intent").with(this@StationHuaweiManagerImpl).print()
                try {
                    Tasks.await(
                        LocationServices.getGeofenceService(mContext).deleteGeofenceList(
                            listOf(
                                GEOFENCE_STATION_ID_LOW,
                                GEOFENCE_STATION_ID_HIGH
                            )
                        ), 10, TimeUnit.SECONDS
                    )
                    Tasks.await(LocationServices.getGeofenceService(mContext).deleteGeofenceList(intent), 10, TimeUnit.SECONDS)
                    true
                } catch (e: Exception) {
                    Logger.e(GeoException.StationUnSubscription(e)).with(this@StationHuaweiManagerImpl).print()
                    false
                } finally {
                    intent.cancel()
                }
            } ?: true
        }
        .doOnSuccess {
            Logger.i().setResult(it).with(this@StationHuaweiManagerImpl).print()
        }
        .ignoreElement()


    private fun getIntent(flags: Int): PendingIntent? {
        Logger.d().print()
        val intent = Intent(mContext, StationHuaweiReceiver::class.java)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        return PendingIntent.getBroadcast(mContext, REQUEST_STATION_HUAWEI, intent, flags)
    }


    companion object {
        private const val REQUEST_STATION_HUAWEI = 6748607
        private const val GEOFENCE_STATION_ID_LOW = "GeoStation#blabla#34563026"
        private const val GEOFENCE_STATION_ID_HIGH = "GeoStation#blabla#75743135"
    }
}