package org.findmykids.geo.data.repository.storage.error

import io.reactivex.Completable
import io.reactivex.Single
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.db.GeoDatabase
import org.findmykids.geo.data.db.factory.ErrorEntityFactory
import org.findmykids.geo.data.db.model.ErrorEntity
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Error
import org.findmykids.geo.data.model.Session
import org.findmykids.geo.data.network.SocketClient
import org.findmykids.geo.data.network.factory.CoordsFactory
import org.findmykids.geo.data.network.model.SocketData
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class ErrorRepositoryImpl @Inject constructor(
    private val mSocketClient: SocketClient,
    private val mGeoDatabase: GeoDatabase
) : ErrorRepository {

    override fun toString(): String = ""

    override fun send(error: Error): Completable = Single
        .defer {
            Logger.d().addArg(error).with(this@ErrorRepositoryImpl).print()
            mSocketClient.send(SocketData(CoordsFactory.createCoords(true, error).toByteArray()))
        }
        .doOnSuccess { isSended ->
            Logger.d().addArg(isSended).with(this@ErrorRepositoryImpl).print()
            if (!isSended) {
                synchronized(mGeoDatabase.lock) {
                    mGeoDatabase
                        .errorsDao()
                        .insert(ErrorEntityFactory.create(error))
                }
            }
        }
        .ignoreElement()
        .doOnComplete {
            Logger.d("Complete").with(this@ErrorRepositoryImpl).print()
        }

    override fun sendAll(session: Session, configuration: Configuration.ErrorStorageConfiguration): Completable = Single
        .fromCallable {
            Logger.d("online").with(this@ErrorRepositoryImpl).print()
            synchronized(mGeoDatabase.lock) {
                mGeoDatabase.errorsDao().selectOnline(session.index, System.currentTimeMillis())
            }
        }
        .flatMap { entities ->
            Logger.d().addArg(entities.size).with(this@ErrorRepositoryImpl).print()
            if (entities.isNotEmpty()) {
                val errors = entities.map { ErrorFactory.create(it) }
                val coords = CoordsFactory.createCoords(true, * errors.toTypedArray())
                mSocketClient
                    .send(SocketData(coords.toByteArray()))
                    .doOnSuccess { isSended ->
                        Logger.d().addArg(isSended).with(this@ErrorRepositoryImpl).print()
                        if (isSended) {
                            removeErrors(entities)
                        }
                    }
            } else {
                Single.just(false)
            }
        }
        .flatMapCompletable {
            val count = mGeoDatabase.errorsDao().getRowCount()
            Logger.d("offline").addArg(count).with(this@ErrorRepositoryImpl).print()
            var isLastSended = false
            Single
                .fromCallable { mGeoDatabase.errorsDao().select(configuration.offlineMaxCount) }
                .flatMap { entities ->
                    Logger.d().addArg(entities.size).with(this@ErrorRepositoryImpl).print()
                    if (entities.isNotEmpty()) {
                        val errors = entities.map { ErrorFactory.create(it) }
                        val coords = CoordsFactory.createCoords(false, * errors.toTypedArray())
                        mSocketClient
                            .send(SocketData(coords.toByteArray()))
                            .map { isSended -> Pair(entities, isSended) }
                    } else {
                        Single.just(Pair(entities, false))
                    }
                }
                .doOnSuccess { (entities, isSended) ->
                    Logger.d().addArg(entities.size).addArg(isSended).with(this@ErrorRepositoryImpl).print()
                    isLastSended = isSended
                    if (isSended) {
                        removeErrors(entities)
                    }
                }
                .delay(if (count > 0) configuration.timeoutOnSendOffline else 100, TimeUnit.MILLISECONDS)
                .repeatUntil {
                    val rowCount = mGeoDatabase.errorsDao().getRowCount()
                    Logger.d("repeat").addArg(rowCount).addArg(isLastSended).with(this@ErrorRepositoryImpl).print()
                    !isLastSended || rowCount == 0
                }
                .ignoreElements()
        }
        .doOnComplete {
            Logger.d("Complete").with(this@ErrorRepositoryImpl).print()
        }

    override fun clear(): Completable = Completable
        .fromCallable {
            Logger.d().with(this@ErrorRepositoryImpl).print()
            synchronized(mGeoDatabase.lock) {
                mGeoDatabase
                    .errorsDao()
                    .deleteAll()
            }
        }
        .doOnComplete {
            Logger.d("Complete").with(this@ErrorRepositoryImpl).print()
        }


    private fun removeErrors(entities: List<ErrorEntity>) {
        Logger.d().addArg(entities.size).print()
        synchronized(mGeoDatabase.lock) {
            entities.forEach {
                mGeoDatabase
                    .errorsDao()
                    .delete(it)
            }
        }
    }
}