package org.findmykids.geo.data.repository.live.wifi

import android.Manifest.permission.*
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.wifi.WifiManager
import android.os.Build
import android.os.Handler
import android.os.HandlerThread
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.Container
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.ManufactureUtil
import org.findmykids.geo.common.utils.WifiUtil
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Wifi
import org.findmykids.geo.data.preferences.LocalPreferences
import java.util.*
import java.util.concurrent.TimeUnit
import javax.inject.Inject
import kotlin.math.max


internal class WifiRepositoryImpl @Inject constructor(
    private val mContext: Context,
    private val mLocalPreferences: LocalPreferences
) : WifiRepository() {

    private var mNextScanDisposable: Disposable? = null
    private var mRestartDisposable: Disposable? = null
    private lateinit var mConfiguration: Configuration.WifiDataConfiguration
    private lateinit var mHandlerThread: HandlerThread

    private val mWifiManager = ContextCompat.getSystemService(mContext, WifiManager::class.java)!!
    private val mConnectivityManager = ContextCompat.getSystemService(mContext, ConnectivityManager::class.java)!!
    private var mWifiStateBeforeStart: Boolean = mWifiManager.isWifiEnabled

    private var mIsWifiScanReceiveRegistered = false
    private val mWifiScanReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            Logger.d().with(this@WifiRepositoryImpl).print()
            if (!isStarted()) {
                return
            }
            stopWifiScan()
            val success = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                intent.getBooleanExtra(WifiManager.EXTRA_RESULTS_UPDATED, false)
            } else {
                true
            }
            if (success) {
                sendEvent(
                    WifisEvent(
                        wifiFullList =
                        WifiListFactory.createFromScanResults(
                            mWifiManager.scanResults,
                            WifiUtil.getCurrentSsid(mConnectivityManager, mWifiManager)
                        )
                    )
                )
                scheduledNextScan(mConfiguration, mHandlerThread)
            } else {
                sendError(GeoException.WifiScanFailure())
                restart(mConfiguration, mHandlerThread)
            }
        }
    }


    override fun toString(): String = ""

    override fun start(configuration: Configuration.WifiDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        mConfiguration = configuration
        mHandlerThread = handlerThread
        scheduledNextScan(configuration, handlerThread)
    }

    override fun stop() {
        Logger.i().print()
        stopWifiScan()
        mNextScanDisposable?.dispose()
        mRestartDisposable?.dispose()
    }

    override fun getCurrentWifi(): Single<Container<Wifi>> = Single
        .fromCallable {
            Logger.d().with(this@WifiRepositoryImpl).print()
            val ssid = WifiUtil.getCurrentSsid(mConnectivityManager, mWifiManager)
            val mac = WifiUtil.getCurrentMac(mWifiManager)
            val level = WifiUtil.getCurrentLevel(mWifiManager)
            val wifi = if (ssid != null && mac != null && level != null) {
                Wifi(ssid, mac, level)
            } else {
                null
            }
            Container(wifi)
        }
        .doOnSuccess {
            Logger.d().setResult(it.value).with(this@WifiRepositoryImpl).print()
        }


    private fun startWifiScan(configuration: Configuration.WifiDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mWifiStateBeforeStart = mWifiManager.isWifiEnabled
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O &&
            mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(CHANGE_WIFI_STATE) != PackageManager.PERMISSION_GRANTED
        ) {
            sendError(GeoException.WifiNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION, CHANGE_WIFI_STATE))
            restart(configuration, handlerThread)
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q &&
            mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(ACCESS_WIFI_STATE) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(CHANGE_WIFI_STATE) != PackageManager.PERMISSION_GRANTED
        ) {
            sendError(GeoException.WifiNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION, ACCESS_WIFI_STATE, CHANGE_WIFI_STATE))
            restart(configuration, handlerThread)
        } else {
            @Suppress("DEPRECATION")
            val isEnabled = when {
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2 && mWifiManager.isScanAlwaysAvailable -> true
                mWifiManager.isWifiEnabled -> true
                ManufactureUtil.isXiaomiNoGoogleOne() -> false
                else -> mWifiManager.setWifiEnabled(true)
            }
            if (isEnabled) {
                mLocalPreferences.setLastWifiScanTime(Date())
                val intentFilter = IntentFilter()
                intentFilter.addAction(WifiManager.SCAN_RESULTS_AVAILABLE_ACTION)
                mIsWifiScanReceiveRegistered = true
                mContext.registerReceiver(mWifiScanReceiver, intentFilter, null, Handler(handlerThread.looper))
                @Suppress("DEPRECATION")
                val success = mWifiManager.startScan()
                if (!success) {
                    sendError(GeoException.WifiScanFailure())
                    stopWifiScan()
                    restart(configuration, handlerThread)
                }
            } else {
                sendError(GeoException.WifiNotEnabled())
                stopWifiScan()
                restart(configuration, handlerThread)
            }
        }
    }

    private fun stopWifiScan() {
        Logger.d().print()
        if (mIsWifiScanReceiveRegistered) {
            mIsWifiScanReceiveRegistered = false
            mContext.unregisterReceiver(mWifiScanReceiver)
        }
        if (mWifiStateBeforeStart != mWifiManager.isWifiEnabled) {
            @Suppress("DEPRECATION")
            mWifiManager.isWifiEnabled = mWifiStateBeforeStart
        }
    }

    private fun scheduledNextScan(configuration: Configuration.WifiDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        val minInterval = if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
            TimeUnit.MINUTES.toMillis(10)
        } else {
            // https://developer.android.com/guide/topics/connectivity/wifi-scan
            TimeUnit.MINUTES.toMillis(30)
        }
        val dif = System.currentTimeMillis() - mLocalPreferences.getLastWifiScanTime().time
        val delay = max(0, minInterval - dif)
        if (delay == 0L) {
            startWifiScan(configuration, handlerThread)
        } else {
            mNextScanDisposable = Completable
                .timer(delay, TimeUnit.MILLISECONDS, Schedulers.newThread())
                .subscribe({
                    mNextScanDisposable = null
                    Logger.d("restart").with(this@WifiRepositoryImpl).print()
                    if (isStarted()) {
                        startWifiScan(configuration, handlerThread)
                    }
                }, {
                    mNextScanDisposable = null
                    sendError(GeoException.WifiSchedule(it))
                    restart(configuration, handlerThread)
                })
        }
    }

    private fun restart(configuration: Configuration.WifiDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mRestartDisposable = Completable
            .timer(configuration.restartDelay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                mRestartDisposable = null
                Logger.d("restart").with(this@WifiRepositoryImpl).print()
                if (isStarted()) {
                    scheduledNextScan(configuration, handlerThread)
                }
            }, {
                mRestartDisposable = null
                sendError(GeoException.WifiRestart(it))
                restart(configuration, handlerThread)
            })
    }
}