package org.findmykids.geo.data.repository.live.location

import android.Manifest
import android.content.Context
import android.location.LocationManager
import android.os.HandlerThread
import androidx.annotation.RequiresPermission
import androidx.core.content.ContextCompat
import com.huawei.hmf.tasks.Tasks
import com.huawei.hms.location.LocationCallback
import com.huawei.hms.location.LocationRequest
import com.huawei.hms.location.LocationResult
import com.huawei.hms.location.LocationServices
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.LocationManagerUtil
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.LocationSource
import java.util.concurrent.TimeUnit


internal class LocationSourceProviderHuaweiFusedImpl(
    private val mContext: Context,
    private val mCallback: LocationSourceProvider.Callback,
    private val mHandlerThread: HandlerThread
) : LocationSourceProvider {

    private var mLocationCallback = object : LocationCallback() {
        override fun onLocationResult(locationResult: LocationResult?) {
            Logger.d().addArg(locationResult).print()
            locationResult ?: return
            for (location in locationResult.locations) {
                mCallback.onLocation(LocationFactory.createLocation(location, LocationSource.HUAWEI_FUSED))
            }
        }
    }


    override fun toString(): String = ""

    @RequiresPermission(anyOf = [Manifest.permission.ACCESS_COARSE_LOCATION, Manifest.permission.ACCESS_FINE_LOCATION])
    override fun start(configuration: Configuration.LocationDataConfiguration) {
        Logger.d().addArg(configuration).print()
        if (LocationManagerUtil.isEnabled(ContextCompat.getSystemService(mContext, LocationManager::class.java)!!)) {
            val locationRequest = LocationRequest()
                .setInterval(configuration.interval)
                .setPriority(configuration.priority)
                .setFastestInterval(configuration.fastestInterval)
                .setMaxWaitTime(configuration.maxWaitTime)
            if (configuration.smallestDisplacement > 0) {
                locationRequest.smallestDisplacement = configuration.smallestDisplacement
            }
            try {
                Tasks.await(
                    LocationServices.getFusedLocationProviderClient(mContext).requestLocationUpdates(
                        locationRequest,
                        mLocationCallback,
                        mHandlerThread.looper
                    ), 10, TimeUnit.SECONDS
                )
            } catch (e: Exception) {
                throw GeoException.HuaweiFusedStart(e)
            }
        } else {
            throw GeoException.HuaweiFusedLocationManagerIsNotEnabled()
        }
    }

    override fun stop() {
        Logger.d().print()
        try {
            Tasks.await(LocationServices.getFusedLocationProviderClient(mContext).removeLocationUpdates(mLocationCallback), 10, TimeUnit.SECONDS)
        } catch (e: Exception) {
            throw GeoException.HuaweiFusedStop(e)
        }
    }
}