package org.findmykids.geo.data.repository.live.location

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.HandlerThread
import com.huawei.hmf.tasks.Tasks
import com.huawei.hms.location.LocationServices
import io.reactivex.Completable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.common.utils.ManufactureUtil
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Location
import org.findmykids.geo.data.preferences.LocalPreferences
import java.util.concurrent.TimeUnit
import javax.inject.Inject


internal class LocationRepositoryImpl @Inject constructor(
    private val mContext: Context,
    private val mLocalPreferences: LocalPreferences
) : LocationRepository() {

    private var mDisposable: Disposable? = null
    private var mIndex: Long = 1L
    private var mLocationCallback = object : LocationSourceProvider.Callback {
        override fun onLocation(location: Location) {
            Logger.d().addArg(location).print()
            sendEvent(LocationEvent(location, mIndex++, mLocalPreferences.getAndIncreaseDefineGlobalIndex()))
        }
    }
    private var mLocationSourceProvider: LocationSourceProvider? = null


    override fun toString(): String = ""

    override fun start(configuration: Configuration.LocationDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        mDisposable?.dispose()
        mDisposable = null
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&
            mContext.checkSelfPermission(ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            mContext.checkSelfPermission(ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
        ) {
            sendError(GeoException.LocationNoPermission(ACCESS_COARSE_LOCATION, ACCESS_FINE_LOCATION))
            restart(configuration, handlerThread)
        } else {
            mLocationSourceProvider = if (ManufactureUtil.isHuawei() && ManufactureUtil.isEmuiVersion9_1_1orAfter()) {
                try {
                    val client = LocationServices.getFusedLocationProviderClient(mContext)!!
                    Tasks.await(client.lastLocation, 10, TimeUnit.SECONDS)
                    LocationSourceProviderHuaweiFusedImpl(mContext, mLocationCallback, handlerThread)
                } catch (e: Exception) {
                    if (ManufactureUtil.isPlayServicesAvailable(mContext)) {
                        LocationSourceProviderGoogleFusedImpl(mContext, mLocationCallback, handlerThread)
                    } else {
                        LocationSourceProviderGpsProviderImpl(mContext, mLocationCallback, handlerThread)
                    }
                }
            } else if (ManufactureUtil.isPlayServicesAvailable(mContext)) {
                LocationSourceProviderGoogleFusedImpl(mContext, mLocationCallback, handlerThread)
            } else {
                LocationSourceProviderGpsProviderImpl(mContext, mLocationCallback, handlerThread)
            }
            try {
                mLocationSourceProvider?.start(configuration)
            } catch (e: GeoException) {
                sendError(e)
                restart(configuration, handlerThread)
            }
        }
    }

    override fun stop() {
        Logger.i().print()
        if (mDisposable == null) {
            try {
                mLocationSourceProvider?.stop()
            } catch (e: GeoException) {
                sendError(e)
            }
            mLocationSourceProvider = null
        } else {
            mDisposable?.dispose()
            mDisposable = null
        }
    }


    private fun restart(configuration: Configuration.LocationDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mDisposable = Completable
            .timer(configuration.restartDelay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                Logger.d("restart").with(this@LocationRepositoryImpl).print()
                mDisposable = null
                if (isStarted()) {
                    start(configuration, handlerThread)
                }
            }, {
                mDisposable = null
                sendError(GeoException.LocationRestart(it))
                restart(configuration, handlerThread)
            })
    }
}