package org.findmykids.geo.data.repository.live.lbs

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.HandlerThread
import android.telephony.CellInfo
import android.telephony.TelephonyManager
import android.telephony.gsm.GsmCellLocation
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import io.reactivex.Completable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import org.findmykids.geo.common.GeoException
import org.findmykids.geo.common.logger.Logger
import org.findmykids.geo.data.model.Configuration
import org.findmykids.geo.data.model.Lbs
import org.findmykids.geo.data.preferences.LocalPreferences
import java.util.*
import java.util.concurrent.TimeUnit
import javax.inject.Inject
import kotlin.math.max


internal class LbsRepositoryImpl @Inject constructor(
    private val mContext: Context,
    private val mLocalPreferences: LocalPreferences
) : LbsRepository() {

    private var mNextScanDisposable: Disposable? = null
    private var mRestartDisposable: Disposable? = null
    private lateinit var mConfiguration: Configuration.LbsDataConfiguration


    private val mCallback: TelephonyManager.CellInfoCallback by lazy {
        @RequiresApi(Build.VERSION_CODES.Q)
        object : TelephonyManager.CellInfoCallback() {
            override fun onCellInfo(cellInfo: MutableList<CellInfo>) {
                Logger.d().with(this@LbsRepositoryImpl).print()
                sendEvent(LbssEvent(LbsListFactory.createFromCellInfoList(cellInfo)))
                mLocalPreferences.setLastLbsScanTime(Date())
                if (isStarted()) {
                    scheduledNextScan(mConfiguration)
                }
            }
        }
    }


    override fun toString(): String = ""

    override fun start(configuration: Configuration.LbsDataConfiguration, handlerThread: HandlerThread) {
        Logger.i().addArg(configuration).print()
        mConfiguration = configuration
//        scheduledNextScan(configuration)
    }

    override fun stop() {
        Logger.i().print()
        mNextScanDisposable?.dispose()
        mRestartDisposable?.dispose()
    }

    private fun scheduledNextScan(configuration: Configuration.LbsDataConfiguration) {
        Logger.d().print()
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            val dif = System.currentTimeMillis() - mLocalPreferences.getLastLbsScanTime().time
            val delay = max(0L, TimeUnit.MINUTES.toMillis(1) - dif)
            if (delay == 0L) {
                startLbsScan(configuration)
            } else {
                mNextScanDisposable = Completable
                    .timer(delay, TimeUnit.MILLISECONDS, Schedulers.newThread())
                    .subscribe({
                        mNextScanDisposable = null
                        Logger.d("restart").with(this@LbsRepositoryImpl).print()
                        if (isStarted()) {
                            startLbsScan(configuration)
                        }
                    }, {
                        mNextScanDisposable = null
                        sendError(GeoException.LbsSchedule(it))
                        restart(configuration)
                    })
            }
        } else {
            startLbsScan(configuration)
        }
    }

    private fun restart(configuration: Configuration.LbsDataConfiguration, handlerThread: HandlerThread) {
        Logger.d().print()
        mRestartDisposable = Completable
            .timer(configuration.restartDelay, TimeUnit.MILLISECONDS, Schedulers.newThread())
            .subscribe({
                mRestartDisposable = null
                Logger.d("restart").with(this@LbsRepositoryImpl).print()
                if (isStarted()) {
                    scheduledNextScan(configuration)
                }
            }, {
                mRestartDisposable = null
                sendError(GeoException.LbsRestart(it))
                restart(configuration)
            })
    }


    private fun startLbsScan(configuration: Configuration.LbsDataConfiguration) {
        Logger.d().print()
        val telephonyManager = ContextCompat.getSystemService(mContext, TelephonyManager::class.java)!!
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&
            mContext.checkSelfPermission(Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED
        ) {
            sendError(GeoException.LbsNoPermission(Manifest.permission.ACCESS_COARSE_LOCATION))
            restart(configuration)
        } else if (mIsLbsWasRequested && Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            // https://developer.android.com/reference/android/telephony/TelephonyManager#getAllCellInfo()
            telephonyManager.requestCellInfoUpdate(mContext.mainExecutor, mCallback)
        } else {
            val lbsList = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                val lbsList = telephonyManager.allCellInfo?.let {
                    LbsListFactory.createFromCellInfoList(it).toMutableList()
                } ?: mutableListOf()
                if (lbsList.isEmpty()) {
                    val networkOperator = telephonyManager.networkOperator?.takeIf { it.length >= 3 }
                    val mcc = networkOperator?.substring(0, 3)?.toIntOrNull()
                    val mnc = networkOperator?.substring(3)?.toIntOrNull()

                    @Suppress("DEPRECATION")
                    val cellLocation = telephonyManager.cellLocation as? GsmCellLocation
                    if (mcc != null && mnc != null && cellLocation != null) {
                        lbsList.add(Lbs(mcc, mnc, cellLocation.cid, cellLocation.lac, -1))
                    }
                }
                lbsList.toList()
            } else {
                listOf()
            }
            mIsLbsWasRequested = true
            mLocalPreferences.setLastLbsScanTime(Date())
            sendEvent(LbssEvent(lbsList))
            scheduledNextScan(configuration)
        }
    }


    companion object {
        private var mIsLbsWasRequested = false
    }
}