package org.findmykids.geo.common.utils

import android.annotation.SuppressLint
import android.net.ConnectivityManager
import android.net.wifi.WifiInfo
import android.net.wifi.WifiManager
import android.text.TextUtils
import org.findmykids.geo.common.logger.Logger
import java.io.*
import java.net.NetworkInterface
import java.util.*


internal object WifiUtil {
    private const val MARSHMALLOW_MAC_ADDRESS = "02:00:00:00:00:00"
    private const val FILE_ADDRESS_MAC = "/sys/class/net/wlan0/address"


    @Suppress("DEPRECATION")
    fun getCurrentLevel(wifiManager: WifiManager): Int? =
        wifiManager.connectionInfo?.rssi

    @Suppress("DEPRECATION")
    fun getCurrentSsid(connectivityManager: ConnectivityManager, wifiManager: WifiManager): String? =
        connectivityManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI)?.let { networkInfo ->
            if (networkInfo.isConnected) {
                val connectionInfo = wifiManager.connectionInfo
                if (connectionInfo != null && !TextUtils.isEmpty(connectionInfo.ssid)) {
                    connectionInfo.ssid.replace("\"$".toRegex(), "").replace("^\"".toRegex(), "")
                } else {
                    null
                }
            } else {
                null
            }
        }

    @SuppressLint("HardwareIds")
    fun getCurrentMac(wifiManager: WifiManager): String? =
        if (wifiManager.isWifiEnabled) {
            val wifiInfo: WifiInfo = wifiManager.connectionInfo
            if (wifiInfo.macAddress == MARSHMALLOW_MAC_ADDRESS) {
                try {
                    getAddressMacByInterface() ?: getAddressMacByFile(wifiManager)
                } catch (e: Exception) {
                    Logger.e(e).print()
                    null
                }
            } else {
                wifiInfo.macAddress
            }
        } else {
            null
        }


    @SuppressLint("DefaultLocale")
    private fun getAddressMacByInterface(): String? {
        try {
            val all: List<NetworkInterface> = Collections.list(NetworkInterface.getNetworkInterfaces())
            for (nif in all) {
                if (nif.name.toLowerCase() == "wlan0") {
                    val macBytes: ByteArray = nif.hardwareAddress ?: return ""
                    val builder = StringBuilder()
                    for (b in macBytes) {
                        builder.append(String.format("%02X:", b))
                    }
                    if (builder.isNotEmpty()) {
                        builder.deleteCharAt(builder.length - 1)
                    }
                    return builder.toString()
                }
            }
        } catch (e: Exception) {
            Logger.e(e).print()
        }
        return null
    }

    @Throws(java.lang.Exception::class)
    private fun getAddressMacByFile(wifiMan: WifiManager): String? =
        if (wifiMan.isWifiEnabled) {
            val file = File(FILE_ADDRESS_MAC)
            val inputStream = FileInputStream(file)
            try {
                val reader = BufferedReader(InputStreamReader(inputStream))
                val builder = StringBuilder()
                var line: String?
                while (reader.readLine().also { line = it } != null) {
                    builder.append(line).append('\n')
                }
                builder.toString()
            } catch (e: IOException) {
                Logger.e(e).print()
                null
            } finally {
                try {
                    inputStream.close()
                } catch (e: IOException) {
                    Logger.e(e).print()
                }
            }
        } else {
            null
        }
}