package org.findmykids.geo.common

import android.telephony.CellInfo
import org.findmykids.geo._todo.Geo
import org.findmykids.geo.data.network.model.SocketCommand
import java.io.PrintStream
import java.io.PrintWriter


sealed class GeoException : Exception() {

    abstract fun getInnerThrowable(): Throwable?


    override fun printStackTrace() {
        super.printStackTrace()
        getInnerThrowable()?.printStackTrace()
    }

    override fun printStackTrace(s: PrintWriter) {
        super.printStackTrace(s)
        getInnerThrowable()?.printStackTrace(s)
    }

    override fun printStackTrace(s: PrintStream) {
        super.printStackTrace(s)
        getInnerThrowable()?.printStackTrace(s)
    }


    // region Managers errors

    class ActivityTransitionNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Activity Transition no permission: ${mPermissions.map { it }}"
    }

    class ActivityTransitionSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Activity Transition subscription error caused by: ${mThrowable.message}"
    }

    class ActivityTransitionUnSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Activity Transition unsubscription error caused by: ${mThrowable.message}"
    }


    class PassiveNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Passive no permission: ${mPermissions.map { it }}"
    }

    class PassiveSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Passive subscription error caused by: ${mThrowable.message}"
    }

    class PassiveUnSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Passive unsubscription error caused by: ${mThrowable.message}"
    }


    class ZoneSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Zone subscription error caused by: ${mThrowable.message}"
    }

    class ZoneUnSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Zone unsubscription error caused by: ${mThrowable.message}"
    }


    class StationSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Station subscription error caused by: ${mThrowable.message}"
    }

    class StationUnSubscription(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Station unsubscription error caused by: ${mThrowable.message}"
    }

    // endregion

    // region Live data errors

    class ActivityNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Activity no permission: ${mPermissions.map { it }}"
    }

    class ActivityStart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Activity subscription error caused by: ${mThrowable.message}"
    }

    class ActivityStop(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Activity unsubscription error caused by: ${mThrowable.message}"
    }

    class ActivityRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Activity restart error caused by: ${mThrowable.message}"
    }


    class GpsNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Gps no permission: ${mPermissions.map { it }}"
    }

    class GpsLocationManagerGpsIsNotEnabled : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Gps provider is not enabled"
    }

    class GpsRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Gps restart error caused by: ${mThrowable.message}"
    }


    class LocationNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Location no permission: ${mPermissions.map { it }}"
    }

    class LocationRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Location restart error caused by: ${mThrowable.message}"
    }


    class GoogleFusedLocationManagerIsNotEnabled : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Google Fused. Location manager is not enabled"
    }

    class GoogleFusedStart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Google Fused start error caused by: ${mThrowable.message}"
    }

    class GoogleFusedStop(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Google Fused stop error caused by: ${mThrowable.message}"
    }


    class HuaweiFusedLocationManagerIsNotEnabled : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Huawei Fused. Location manager is not enabled"
    }

    class HuaweiFusedStart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Huawei Fused start error caused by: ${mThrowable.message}"
    }

    class HuaweiFusedStop(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Huawei Fused stop error caused by: ${mThrowable.message}"
    }


    class GpsProviderLocationManagerGpsIsNotEnabled : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "GPS Provider. Gps provider is not enabled"
    }


    class SensorNoValidSensors : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "No valid sensors"
    }


    class TimeoutStart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Timeout start error caused by: ${mThrowable.message}"
    }

    class TimeoutRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Timeout restart error caused by: ${mThrowable.message}"
    }


    class WifiNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "WifiFull no permission: ${mPermissions.map { it }}"
    }

    class WifiNotEnabled : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "WifiFull not enabled"
    }

    class WifiScanFailure : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "WifiFull scan failed"
    }

    class WifiSchedule(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "WifiFull schedule error caused by: ${mThrowable.message}"
    }

    class WifiRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "WifiFull restart error caused by: ${mThrowable.message}"
    }


    class LbsNoPermission(private vararg val mPermissions: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Lbs no permission: ${mPermissions.map { it }}"
    }

    class LbsSchedule(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Lbs schedule error caused by: ${mThrowable.message}"
    }

    class LbsRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Lbs restart error caused by: ${mThrowable.message}"
    }

    class InvalidLbsCellInfoType(private val mCellInfo: CellInfo) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid lbs cell info: $mCellInfo"
    }


    class InvalidSocketData(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Invalid socket data: ${mThrowable.message}"
    }

    class InvalidSocketEvent(private val mCommand: SocketCommand) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid socket event: ${mCommand.name}"
    }

    class Socket(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Socket error caused by: ${mThrowable.message}"
    }

    class SocketRestart(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Socket restart error caused by: ${mThrowable.message}"
    }

    class SocketClose(private val mStatus: Int, private val mMessage: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Socket closed by status $mStatus and message $mMessage"
    }

    // endregion



    // region Live event interactor

    class InvalidLocatorGeo(private val mGeo: Geo) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid locator geo: $mGeo"
    }

    class EmptyLocation : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Failed to determine location"
    }

    // endregion

    // region DB converter error

    class InvalidGeoType(private val mType: Int) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid geo type: $mType"
    }

    class InvalidConfigurationType(private val mType: Int) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid configuration type: $mType"
    }

    class InvalidCommandType(private val mType: Int) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid command type: $mType"
    }

    // endregion

    // region Yandex locator

    class YandexEmptyData : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "No data for send to Yandex locator"
    }

    class YandexLocatorConnection(private val mThrowable: Throwable) : GeoException() {
        override fun getInnerThrowable(): Throwable? = mThrowable
        override fun toString() = "Yandex locator connection error caused by: ${mThrowable.message}"
    }

    class YandexLocatorResponse(private val mCode: Int, private val mMessage: String) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Yandex locator error code $mCode and message $mMessage"
    }

    // endregion


    class NotSupportedAction(private val mAction: String?) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Invalid action: $mAction"
    }

    class Api(private val mMessage: String?) : GeoException() {
        override fun getInnerThrowable(): Throwable? = null
        override fun toString() = "Api error caused by: $mMessage"
    }
}