// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.checkerboard.alt.javafx.impls;

import org.refcodes.checkerboard.ChangePositionEvent;
import org.refcodes.checkerboard.Checkerboard;
import org.refcodes.checkerboard.ChessmanStatus;
import org.refcodes.checkerboard.Player;
import org.refcodes.checkerboard.impls.CheckerboardImpl;
import org.refcodes.checkerboard.impls.PlayerImpl;
import org.refcodes.component.InitializeException;
import org.refcodes.controlflow.ControlFlowUtility;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.VetoException;
import org.refcodes.graphical.GridMode;
import org.refcodes.graphical.MoveMode;
import org.refcodes.graphical.ScaleMode;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;

import javafx.application.Application;
import javafx.scene.paint.Color;
import javafx.stage.Stage;

public class CheckerboardDemo extends Application {

	private static final double SCALE_FACTOR = 0.75;

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private FxCheckerboardViewerImpl<ChessmanStatus> _checkerboardViewer = null;
	private Checkerboard<ChessmanStatus> _checkerboard = null;

	private Player<ChessmanStatus> _blackRookA = new PlayerImpl<ChessmanStatus>( 0, 0 ).withState( ChessmanStatus.BLACK_ROOK );
	private Player<ChessmanStatus> _blackKnightB = new PlayerImpl<ChessmanStatus>( 1, 0 ).withState( ChessmanStatus.BLACK_KNIGHT );
	private Player<ChessmanStatus> _blackBishopC = new PlayerImpl<ChessmanStatus>( 2, 0 ).withState( ChessmanStatus.BLACK_BISHOP );
	private Player<ChessmanStatus> _blackQueen = new PlayerImpl<ChessmanStatus>( 3, 0 ).withState( ChessmanStatus.BLACK_QUEEN );
	private Player<ChessmanStatus> _blackKing = new PlayerImpl<ChessmanStatus>( 4, 0 ).withState( ChessmanStatus.BLACK_KING );
	private Player<ChessmanStatus> _blackBishopF = new PlayerImpl<ChessmanStatus>( 5, 0 ).withState( ChessmanStatus.BLACK_BISHOP );
	private Player<ChessmanStatus> _blackKnightG = new PlayerImpl<ChessmanStatus>( 6, 0 ).withState( ChessmanStatus.BLACK_KNIGHT );
	private Player<ChessmanStatus> _blackRookH = new PlayerImpl<ChessmanStatus>( 7, 0 ).withState( ChessmanStatus.BLACK_ROOK );
	private Player<ChessmanStatus> _blackPawnA = new PlayerImpl<ChessmanStatus>( 0, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnB = new PlayerImpl<ChessmanStatus>( 1, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnC = new PlayerImpl<ChessmanStatus>( 2, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnD = new PlayerImpl<ChessmanStatus>( 3, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnE = new PlayerImpl<ChessmanStatus>( 4, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnF = new PlayerImpl<ChessmanStatus>( 5, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnG = new PlayerImpl<ChessmanStatus>( 6, 1 ).withState( ChessmanStatus.BLACK_PAWN );
	private Player<ChessmanStatus> _blackPawnH = new PlayerImpl<ChessmanStatus>( 7, 1 ).withState( ChessmanStatus.BLACK_PAWN );

	private Player<ChessmanStatus> _whiteRookA = new PlayerImpl<ChessmanStatus>( 0, 7 ).withState( ChessmanStatus.WHITE_ROOK );
	private Player<ChessmanStatus> _whiteKnightB = new PlayerImpl<ChessmanStatus>( 1, 7 ).withState( ChessmanStatus.WHITE_KNIGHT );
	private Player<ChessmanStatus> _whiteBishopC = new PlayerImpl<ChessmanStatus>( 2, 7 ).withState( ChessmanStatus.WHITE_BISHOP );
	private Player<ChessmanStatus> _whiteQueen = new PlayerImpl<ChessmanStatus>( 3, 7 ).withState( ChessmanStatus.WHITE_QUEEN );
	private Player<ChessmanStatus> _whiteKing = new PlayerImpl<ChessmanStatus>( 4, 7 ).withState( ChessmanStatus.WHITE_KING );
	private Player<ChessmanStatus> _whiteBishopF = new PlayerImpl<ChessmanStatus>( 5, 7 ).withState( ChessmanStatus.WHITE_BISHOP );
	private Player<ChessmanStatus> _whiteKnightG = new PlayerImpl<ChessmanStatus>( 6, 7 ).withState( ChessmanStatus.WHITE_KNIGHT );
	private Player<ChessmanStatus> _whiteRookH = new PlayerImpl<ChessmanStatus>( 7, 7 ).withState( ChessmanStatus.WHITE_ROOK );
	private Player<ChessmanStatus> _whitePawnA = new PlayerImpl<ChessmanStatus>( 0, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnB = new PlayerImpl<ChessmanStatus>( 1, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnC = new PlayerImpl<ChessmanStatus>( 2, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnD = new PlayerImpl<ChessmanStatus>( 3, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnE = new PlayerImpl<ChessmanStatus>( 4, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnF = new PlayerImpl<ChessmanStatus>( 5, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnG = new PlayerImpl<ChessmanStatus>( 6, 6 ).withState( ChessmanStatus.WHITE_PAWN );
	private Player<ChessmanStatus> _whitePawnH = new PlayerImpl<ChessmanStatus>( 7, 6 ).withState( ChessmanStatus.WHITE_PAWN );

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	private void initializePlayer( Checkerboard<ChessmanStatus> aCheckerboard ) {
		LOGGER.info( "Initializing players on checkerboard ..." );
		aCheckerboard.putPlayer( _blackRookA );
		aCheckerboard.putPlayer( _blackKnightB );
		aCheckerboard.putPlayer( _blackBishopC );
		aCheckerboard.putPlayer( _blackQueen );
		aCheckerboard.putPlayer( _blackKing.withHide().withState( ChessmanStatus.BLACK_KING_KO ) );
		aCheckerboard.putPlayer( _blackBishopF );
		aCheckerboard.putPlayer( _blackKnightG );
		aCheckerboard.putPlayer( _blackRookH );
		aCheckerboard.putPlayer( _blackPawnA );
		aCheckerboard.putPlayer( _blackPawnB );
		aCheckerboard.putPlayer( _blackPawnC );
		aCheckerboard.putPlayer( _blackPawnD );
		aCheckerboard.putPlayer( _blackPawnE );
		aCheckerboard.putPlayer( _blackPawnF );
		aCheckerboard.putPlayer( _blackPawnG );
		aCheckerboard.putPlayer( _blackPawnH );

		aCheckerboard.putPlayer( _whiteRookA );
		aCheckerboard.putPlayer( _whiteKnightB );
		aCheckerboard.putPlayer( _whiteBishopC );
		aCheckerboard.putPlayer( _whiteQueen );
		aCheckerboard.putPlayer( _whiteKing );
		aCheckerboard.putPlayer( _whiteBishopF );
		aCheckerboard.putPlayer( _whiteKnightG );
		aCheckerboard.putPlayer( _whiteRookH );
		aCheckerboard.putPlayer( _whitePawnA );
		aCheckerboard.putPlayer( _whitePawnB );
		aCheckerboard.putPlayer( _whitePawnC );
		aCheckerboard.putPlayer( _whitePawnD );
		aCheckerboard.putPlayer( _whitePawnE );
		aCheckerboard.putPlayer( _whitePawnF );
		aCheckerboard.putPlayer( _whitePawnG );
		aCheckerboard.putPlayer( _whitePawnH );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Anchor for JavaFx to start the application
	 * 
	 * @throws InitializeException
	 */
	@Override
	public void start( Stage aPrimaryStage ) {
		try {
			aPrimaryStage.setTitle( "Checkerboard" );

			_checkerboard = new CheckerboardImpl<ChessmanStatus>();
			_checkerboardViewer = new FxCheckerboardViewerImpl<ChessmanStatus>( _checkerboard) {
				@Override
				public void onChangePositionEvent( ChangePositionEvent<ChessmanStatus> aEvent, Checkerboard<ChessmanStatus> aSource ) throws VetoException {
					if ( aEvent.getPrecedingPosition().getPositionX() == 0 && aEvent.getPrecedingPosition().getPositionY() == 0 ) { throw new VetoException( "Player <" + aEvent.getSource() + "> is glued to position (1, 1)." ); }
				}
			};

			_checkerboard.withGridDimension( 20, 20 );
			_checkerboard.withGridMode( GridMode.CLOSED );
			_checkerboardViewer.withMoveMode( MoveMode.SMOOTH ).withScaleMode( ScaleMode.SCALE_GRID );
			_checkerboardViewer.withSpriteFactory( new FxChessmenFactoryImpl().withScaleFactor( SCALE_FACTOR ) );
			_checkerboardViewer.withViewportDimension( 16, 16 ).withFieldDimension( 32, 32 ).withFieldGap( 2 ).withMinViewportDimension( 8, 8 );
			_checkerboardViewer.withBackgroundFactory( new FxChessboardFactoryImpl<ChessmanStatus>().withFieldGapColor( Color.rgb( 0x5f, 0xff, 0x00 ) ).withOddFieldColor( Color.rgb( 0x00, 0x5f, 0x00 ) ).withEvenFieldColor( Color.rgb( 0x00, 0x87, 0x00 ) ) );
			_checkerboardViewer.withInitialize( aPrimaryStage ).show();
			_checkerboardViewer.withViewportOffset( 0, 0 );

			ControlFlowUtility.createExecutorService().execute( new Runnable() {
				@Override
				public void run() {
					try {
						
						initializePlayer( _checkerboard );
						_blackRookA.blink();
						changeOffset();
						movePlayer();
						showPlayer();
						resizeGrid();
						_blackKing.blink();
						_blackKing.show();
						removePlayer();
						dragPlayer();
					}
					catch ( Exception ignore ) {
						LOGGER.error( ExceptionUtility.toMessage( ignore ), ignore );
					}
				}
			} );
		}
		catch ( InitializeException e ) {
			LOGGER.error( ExceptionUtility.toMessage( e ), e );
			System.exit( 1 );
		}
	}

	private void changeOffset() {
		_checkerboardViewer.setViewportOffset( _checkerboardViewer.getViewportOffsetX() -2, _checkerboardViewer.getViewportOffsetY() -1 );
	}

	private void movePlayer() throws InterruptedException {
		_whitePawnA.setPosition( _whitePawnA.getPositionX(), _whitePawnA.getPositionY() - 2 );
		Thread.sleep( 500 );
		_blackPawnH.setPosition( _blackPawnH.getPositionX(), _blackPawnH.getPositionY() + 2 );
		Thread.sleep( 500 );
		_whitePawnB.setPosition( _whitePawnB.getPositionX(), _whitePawnB.getPositionY() - 1 );
		Thread.sleep( 500 );
		_blackKnightB.setPosition( _blackKnightB.getPositionX() + 1, _blackKnightB.getPositionY() + 2 );
		Thread.sleep( 500 );
		_whitePawnC.setPosition( _whitePawnC.getPositionX(), _whitePawnC.getPositionY() - 2 );
	}

	private void resizeGrid() throws InitializeException, InterruptedException {
		_checkerboardViewer.withViewportWidth( _checkerboardViewer.getViewportWidth() + 2 );
		Thread.sleep( 500 );
		_checkerboardViewer.withViewportHeight( _checkerboardViewer.getViewportHeight() + 2 );
		Thread.sleep( 500 );
		_checkerboardViewer.withViewportWidth( _checkerboardViewer.getViewportWidth() + 2 );
		Thread.sleep( 500 );
		_checkerboardViewer.withViewportWidth( _checkerboardViewer.getViewportWidth() - 1 );
		Thread.sleep( 500 );
		_checkerboardViewer.withViewportWidth( _checkerboardViewer.getViewportWidth() - 1 );
		Thread.sleep( 500 );
		_checkerboardViewer.withViewportWidth( _checkerboardViewer.getViewportWidth() - 2 );
		_checkerboardViewer.withViewportHeight( _checkerboardViewer.getViewportHeight() + 1 );
		Thread.sleep( 500 );
		_checkerboardViewer.withViewportHeight( _checkerboardViewer.getViewportHeight() - 3 );
	}

	private void removePlayer() throws InterruptedException {
		_checkerboard.removePlayer( _whitePawnA );
		Thread.sleep( 500 );
		_checkerboard.removePlayer( _blackPawnH );
		Thread.sleep( 500 );
		_checkerboard.removePlayer( _whitePawnB );
		Thread.sleep( 500 );
		_checkerboard.removePlayer( _blackKnightB );
		Thread.sleep( 500 );
		_checkerboard.removePlayer( _whitePawnC );
	}

	private void showPlayer() throws InterruptedException {
		_whitePawnA.blink();
	}

	private void dragPlayer() throws InterruptedException {
		_blackKing.blink();
		_blackKing.setState( ChessmanStatus.BLACK_KING );
		_blackKing.draggable();
		_whiteKing.blink();
		_whiteKing.setState( ChessmanStatus.WHITE_KING_KO );
		_whiteKing.stationary();
	}

	/**
	 * Anchor for JavaFx to start the application
	 */
	@Override
	public void stop() {
		_checkerboardViewer.destroy();
	}

	// /////////////////////////////////////////////////////////////////////////
	// MAIN:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Usually ignored, only used if JavaFx can not not be started with
	 * start(Stage)
	 * 
	 * @param args
	 */
	public static void main( String[] args ) {
		launch( args );
	}
}
