// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.checkerboard.alt.javafx.impls;

import org.refcodes.checkerboard.Checkerboard;
import org.refcodes.checkerboard.Player;
import org.refcodes.checkerboard.alt.javafx.impls.FxBoulderDashFactoryImpl;
import org.refcodes.checkerboard.alt.javafx.impls.FxChessboardFactoryImpl;
import org.refcodes.checkerboard.BoulderDashStatus;
import org.refcodes.checkerboard.ChangePositionEvent;
import org.refcodes.checkerboard.impls.CheckerboardImpl;
import org.refcodes.checkerboard.impls.PlayerImpl;
import org.refcodes.component.InitializeException;
import org.refcodes.controlflow.ControlFlowUtility;
import org.refcodes.exception.VetoException;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.graphical.GridMode;
import org.refcodes.graphical.MoveMode;
import org.refcodes.graphical.ScaleMode;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;

import javafx.application.Application;
import javafx.scene.paint.Color;
import javafx.stage.Stage;

public class BoulderDashDemo extends Application {

	private static final double SCALE_FACTOR = 1;

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.getInstance().createInstance();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private FxCheckerboardViewerImpl<BoulderDashStatus> _boulderDashViewer = null;
	private Checkerboard<BoulderDashStatus> _boulderDash = null;

	private Player<BoulderDashStatus> _blackRookA = new PlayerImpl<BoulderDashStatus>( 0, 0 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _blackKnightB = new PlayerImpl<BoulderDashStatus>( 1, 0 ).withState( BoulderDashStatus.STEEL_WALL );
	private Player<BoulderDashStatus> _blackKing = new PlayerImpl<BoulderDashStatus>( 4, 0 ).withState( BoulderDashStatus.ROCKFORD_FACING_FORWARD );
	private Player<BoulderDashStatus> _blackKnightG = new PlayerImpl<BoulderDashStatus>( 6, 0 ).withState( BoulderDashStatus.STEEL_WALL );
	private Player<BoulderDashStatus> _blackRookH = new PlayerImpl<BoulderDashStatus>( 7, 0 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whiteRookA = new PlayerImpl<BoulderDashStatus>( 0, 7 ).withState( BoulderDashStatus.ROCKFORD_FACING_LEFT );
	private Player<BoulderDashStatus> _whiteRookH = new PlayerImpl<BoulderDashStatus>( 7, 7 ).withState( BoulderDashStatus.ROCKFORD_FACING_FORWARD );
	private Player<BoulderDashStatus> _whitePawnA = new PlayerImpl<BoulderDashStatus>( 0, 6 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whitePawnB = new PlayerImpl<BoulderDashStatus>( 1, 6 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whitePawnC = new PlayerImpl<BoulderDashStatus>( 2, 6 ).withState( BoulderDashStatus.DIAMOND );
	private Player<BoulderDashStatus> _whitePawnD = new PlayerImpl<BoulderDashStatus>( 3, 6 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whitePawnE = new PlayerImpl<BoulderDashStatus>( 4, 6 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whitePawnF = new PlayerImpl<BoulderDashStatus>( 5, 6 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whitePawnG = new PlayerImpl<BoulderDashStatus>( 6, 6 ).withState( BoulderDashStatus.BOULDER );
	private Player<BoulderDashStatus> _whitePawnH = new PlayerImpl<BoulderDashStatus>( 7, 6 ).withState( BoulderDashStatus.BOULDER );

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	private void initializePlayer( Checkerboard<BoulderDashStatus> aCheckerboard ) {
		LOGGER.info( "Initializing players on boulderDash ..." );
		aCheckerboard.putPlayer( _blackRookA );
		aCheckerboard.putPlayer( _blackKnightB );
		aCheckerboard.putPlayer( _blackKnightG );
		aCheckerboard.putPlayer( _blackRookH );

		aCheckerboard.putPlayer( _whiteRookA );
		aCheckerboard.putPlayer( _whiteRookH );
		aCheckerboard.putPlayer( _whitePawnA );
		aCheckerboard.putPlayer( _whitePawnB );
		aCheckerboard.putPlayer( _whitePawnC );
		aCheckerboard.putPlayer( _whitePawnD );
		aCheckerboard.putPlayer( _whitePawnE );
		aCheckerboard.putPlayer( _whitePawnF );
		aCheckerboard.putPlayer( _whitePawnG );
		aCheckerboard.putPlayer( _whitePawnH );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Anchor for JavaFx to start the application
	 * 
	 * @throws InitializeException
	 */
	@Override
	public void start( Stage aPrimaryStage ) {
		try {
			aPrimaryStage.setTitle( "Checkerboard" );

			_boulderDash = new CheckerboardImpl<BoulderDashStatus>();
			_boulderDashViewer = new FxCheckerboardViewerImpl<BoulderDashStatus>( _boulderDash) {
				@Override
				public void onChangePositionEvent( ChangePositionEvent<BoulderDashStatus> aEvent, Checkerboard<BoulderDashStatus> aSource ) throws VetoException {
					if ( aEvent.getPrecedingPosition().getPositionX() == 0 && aEvent.getPrecedingPosition().getPositionY() == 0 ) { throw new VetoException( "Player <" + aEvent.getSource() + "> is glued to position (1, 1)." ); }
				}
			};

			_boulderDash.withGridDimension( 20, 20 );
			_boulderDash.withGridMode( GridMode.CLOSED );
			_boulderDashViewer.withMoveMode( MoveMode.SMOOTH ).withScaleMode( ScaleMode.SCALE_GRID );
			_boulderDashViewer.withSpriteFactory( new FxBoulderDashFactoryImpl().withScaleFactor( SCALE_FACTOR ) );
			_boulderDashViewer.withViewportDimension( 16, 16 ).withFieldDimension( 32, 32 ).withFieldGap( 2 ).withMinViewportDimension( 8, 8 );
			_boulderDashViewer.withBackgroundFactory( new FxChessboardFactoryImpl<BoulderDashStatus>().withFieldGapColor( Color.rgb( 0x5f, 0xff, 0x00 ) ).withOddFieldColor( Color.rgb( 0x00, 0x5f, 0x00 ) ).withEvenFieldColor( Color.rgb( 0x00, 0x87, 0x00 ) ) );
			_boulderDashViewer.withInitialize( aPrimaryStage ).show();
			_boulderDashViewer.withViewportOffset( 0, 0 );

			ControlFlowUtility.createExecutorService().execute( new Runnable() {
				@Override
				public void run() {
					try {

						initializePlayer( _boulderDash );
						_blackRookA.blink();
						changeOffset();
						movePlayer();
						showPlayer();
						resizeGrid();
						_blackKing.blink();
						_blackKing.show();
						removePlayer();
						dragPlayer();
					}
					catch ( Exception ignore ) {
						LOGGER.error( ExceptionUtility.toMessage( ignore ), ignore );
					}
				}
			} );
		}
		catch ( InitializeException e ) {
			LOGGER.error( ExceptionUtility.toMessage( e ), e );
			System.exit( 1 );
		}
	}

	private void changeOffset() {
		_boulderDashViewer.setViewportOffset( _boulderDashViewer.getViewportOffsetX() - 2, _boulderDashViewer.getViewportOffsetY() - 1 );
	}

	private void movePlayer() throws InterruptedException {
		_whitePawnA.setPosition( _whitePawnA.getPositionX(), _whitePawnA.getPositionY() - 2 );
		Thread.sleep( 500 );
		Thread.sleep( 500 );
		_whitePawnB.setPosition( _whitePawnB.getPositionX(), _whitePawnB.getPositionY() - 1 );
		Thread.sleep( 500 );
		_blackKnightB.setPosition( _blackKnightB.getPositionX() + 1, _blackKnightB.getPositionY() + 2 );
		Thread.sleep( 500 );
		_whitePawnC.setPosition( _whitePawnC.getPositionX(), _whitePawnC.getPositionY() - 2 );
	}

	private void resizeGrid() throws InitializeException, InterruptedException {
		_boulderDashViewer.withViewportWidth( _boulderDashViewer.getViewportWidth() + 2 );
		Thread.sleep( 500 );
		_boulderDashViewer.withViewportHeight( _boulderDashViewer.getViewportHeight() + 2 );
		Thread.sleep( 500 );
		_boulderDashViewer.withViewportWidth( _boulderDashViewer.getViewportWidth() + 2 );
		Thread.sleep( 500 );
		_boulderDashViewer.withViewportWidth( _boulderDashViewer.getViewportWidth() - 1 );
		Thread.sleep( 500 );
		_boulderDashViewer.withViewportWidth( _boulderDashViewer.getViewportWidth() - 1 );
		Thread.sleep( 500 );
		_boulderDashViewer.withViewportWidth( _boulderDashViewer.getViewportWidth() - 2 );
		_boulderDashViewer.withViewportHeight( _boulderDashViewer.getViewportHeight() + 1 );
		Thread.sleep( 500 );
		_boulderDashViewer.withViewportHeight( _boulderDashViewer.getViewportHeight() - 3 );
	}

	private void removePlayer() throws InterruptedException {
		_boulderDash.removePlayer( _whitePawnA );
		Thread.sleep( 500 );
		_boulderDash.removePlayer( _whitePawnB );
		Thread.sleep( 500 );
		_boulderDash.removePlayer( _blackKnightB );
		Thread.sleep( 500 );
		_boulderDash.removePlayer( _whitePawnC );
	}

	private void showPlayer() throws InterruptedException {
		_whitePawnA.blink();
	}

	private void dragPlayer() throws InterruptedException {
		_blackKing.blink();
		_blackKing.setState( BoulderDashStatus.ROCKFORD_FACING_FORWARD );
		_blackKing.draggable();
	}

	/**
	 * Anchor for JavaFx to start the application
	 */
	@Override
	public void stop() {
		_boulderDashViewer.destroy();
	}

	// /////////////////////////////////////////////////////////////////////////
	// MAIN:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Usually ignored, only used if JavaFx can not not be started with
	 * start(Stage)
	 * 
	 * @param args
	 */
	public static void main( String[] args ) {
		launch( args );
	}
}
