/**
 * Copyright (c) 2013-2024 Nikita Koksharov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.redisson.api;

import java.util.List;
import org.redisson.api.vector.VectorAddArgs;
import org.redisson.api.vector.VectorInfo;
import org.redisson.api.vector.VectorSimilarArgs;
import org.redisson.client.protocol.ScoredEntry;
import reactor.core.publisher.Mono;

/**
 * Reactive interface for Redis Vector Set
 *
 * @author Nikita Koksharov
 *
 */
public interface RVectorSetReactive extends RExpirableReactive {
    /**
     * Adds an element to the vector set
     *
     * @param args vector add arguments
     * @return <code>true</code> if element was added and <code>false</code> if updated
     */
    Mono<Boolean> add(VectorAddArgs args);

    /**
     * Returns the number of elements in the vector set
     *
     * @return cardinality of the vector set
     */
    Mono<Integer> size();

    /**
     * Returns the dimension of vectors in the vector set
     *
     * @return dimension count
     */
    Mono<Integer> dimensions();

    /**
     * Retrieves the vector associated with an element
     *
     * @param element element name
     * @return list of vector coordinates
     */
    Mono<List<Double>> getVector(String element);

    /**
     * Retrieves the raw vector associated with an element
     *
     * @param element element name
     * @return list of raw vector values
     */
    Mono<List<Object>> getRawVector(String element);

    /**
     * Retrieves attributes of a vector set element
     *
     * @param element element name
     * @param clazz class type for deserialization
     * @return deserialized attributes object
     */
    <T> Mono<T> getAttributes(String element, Class<T> clazz);

    /**
     * Returns metadata and internal details about the vector set
     *
     * @return vector set information
     */
    Mono<VectorInfo> getInfo();

    /**
     * Retrieves the neighbors of a specified element
     *
     * @param element element name
     * @return list of neighbor element names
     */
    Mono<List<String>> getNeighbors(String element);

    /**
     * Retrieves the neighbors of a specified element with scores
     *
     * @param element element name
     * @return list of neighbor elements with scores
     */
    Mono<List<ScoredEntry<String>>> getNeighborEntries(String element);

    /**
     * Returns a random element from the vector set
     *
     * @return random element name
     */
    Mono<String> random();

    /**
     * Returns multiple random elements from the vector set
     *
     * @param count number of elements to return
     * @return list of random element names
     */
    Mono<List<String>> random(int count);

    /**
     * Removes an element from the vector set
     *
     * @param element element name to remove
     * @return <code>true</code> if element was removed, false otherwise
     */
    Mono<Boolean> remove(String element);

    /**
     * Sets attributes for a vector set element
     *
     * @param element element name
     * @param attributes attributes object to serialize
     * @return <code>true</code> if attributes were set, false otherwise
     */
    Mono<Boolean> setAttributes(String element, Object attributes);

    /**
     * Retrieves elements similar to a given vector or element
     *
     * @param args vector similarity arguments
     * @return list of similar element names
     */
    Mono<List<String>> getSimilar(VectorSimilarArgs args);

    /**
     * Retrieves elements similar to a given vector or element with scores
     *
     * @param args vector similarity arguments
     * @return list of similar elements with scores
     */
    Mono<List<ScoredEntry<String>>> getSimilarEntries(VectorSimilarArgs args);
}
