/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.cluster;

import io.netty.buffer.ByteBuf;
import io.netty.util.Timeout;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SplittableRandom;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.redisson.api.NodeType;
import org.redisson.api.RFuture;
import org.redisson.client.RedisClient;
import org.redisson.client.RedisClientConfig;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.RedisException;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.client.protocol.RedisStrictCommand;
import org.redisson.cluster.ClusterNodeInfo;
import org.redisson.cluster.ClusterPartition;
import org.redisson.config.BaseMasterSlaveServersConfig;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.Config;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.config.ReadMode;
import org.redisson.config.ShardedSubscriptionMode;
import org.redisson.connection.CRC16;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.MasterSlaveConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.redisson.connection.SingleEntry;
import org.redisson.misc.RedisURI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterConnectionManager
extends MasterSlaveConnectionManager {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final Map<Integer, ClusterPartition> lastPartitions = new ConcurrentHashMap<Integer, ClusterPartition>();
    private final Map<RedisURI, ClusterPartition> lastUri2Partition = new ConcurrentHashMap<RedisURI, ClusterPartition>();
    private volatile Timeout monitorFuture;
    private volatile RedisURI lastClusterNode;
    private RedisStrictCommand<List<ClusterNodeInfo>> clusterNodesCommand;
    private String configEndpointHostName;
    private final AtomicReferenceArray<MasterSlaveEntry> slot2entry = new AtomicReferenceArray(16384);
    private final Map<RedisClient, MasterSlaveEntry> client2entry = new ConcurrentHashMap<RedisClient, MasterSlaveEntry>();
    private ClusterServersConfig cfg;
    private final long seed = ThreadLocalRandom.current().nextLong();

    public ClusterConnectionManager(ClusterServersConfig cfg, Config configCopy) {
        super(cfg, configCopy);
        this.serviceManager.setNatMapper(cfg.getNatMapper());
    }

    @Override
    protected MasterSlaveServersConfig create(BaseMasterSlaveServersConfig<?> cfg) {
        this.cfg = (ClusterServersConfig)cfg;
        return super.create(cfg);
    }

    @Override
    public void doConnect(Set<RedisURI> disconnectedSlaves, Function<RedisURI, String> hostnameMapper) {
        if (this.cfg.getNodeAddresses().isEmpty()) {
            throw new IllegalArgumentException("At least one cluster node should be defined!");
        }
        Throwable lastException = null;
        ArrayList<String> failedMasters = new ArrayList<String>();
        for (String address : this.cfg.getNodeAddresses()) {
            RedisURI addr = new RedisURI(address);
            CompletionStage<RedisConnection> connectionFuture = this.connectToNode(this.cfg, addr, addr.getHost());
            try {
                Collection<ClusterPartition> partitions;
                RedisConnection connection = connectionFuture.toCompletableFuture().get(this.config.getConnectTimeout(), TimeUnit.MILLISECONDS);
                if (this.cfg.getNodeAddresses().size() == 1 && !addr.isIP()) {
                    this.configEndpointHostName = addr.getHost();
                }
                this.clusterNodesCommand = RedisCommands.CLUSTER_NODES;
                if (addr.isSsl()) {
                    this.clusterNodesCommand = RedisCommands.CLUSTER_NODES_SSL;
                }
                List<ClusterNodeInfo> nodes = connection.sync(this.clusterNodesCommand, new Object[0]);
                StringBuilder nodesValue = new StringBuilder();
                for (ClusterNodeInfo clusterNodeInfo : nodes) {
                    nodesValue.append(clusterNodeInfo.getNodeInfo()).append("\n");
                }
                this.log.info("Redis cluster nodes configuration got from {}:\n{}", (Object)connection.getRedisClient().getAddr(), (Object)nodesValue);
                this.lastClusterNode = addr;
                CompletableFuture<Collection<ClusterPartition>> partitionsFuture = this.parsePartitions(nodes);
                try {
                    partitions = partitionsFuture.join();
                }
                catch (CompletionException e) {
                    lastException = e.getCause();
                    break;
                }
                ArrayList<CompletableFuture<Void>> masterFutures = new ArrayList<CompletableFuture<Void>>();
                for (ClusterPartition partition : partitions) {
                    if (partition.isMasterFail()) {
                        failedMasters.add(partition.getMasterAddress().toString());
                        continue;
                    }
                    if (partition.getMasterAddress() == null) {
                        throw new IllegalStateException("Master node: " + partition.getNodeId() + " doesn't have an address.");
                    }
                    CompletionStage<Void> masterFuture = this.addMasterEntry(partition, this.cfg);
                    masterFutures.add(masterFuture.toCompletableFuture());
                }
                CompletableFuture<Void> masterFuture = CompletableFuture.allOf(masterFutures.toArray(new CompletableFuture[0]));
                try {
                    masterFuture.join();
                }
                catch (CompletionException e) {
                    lastException = e.getCause();
                }
                break;
            }
            catch (Exception e) {
                if (e instanceof CompletionException) {
                    e = (Exception)e.getCause();
                }
                lastException = e;
                this.log.warn(e.getMessage());
            }
        }
        if (this.lastPartitions.isEmpty()) {
            this.internalShutdown();
            if (failedMasters.isEmpty()) {
                throw new RedisConnectionException("Can't connect to servers!", lastException);
            }
            throw new RedisConnectionException("Can't connect to servers! Failed masters according to cluster status: " + failedMasters, lastException);
        }
        if (this.cfg.isCheckSlotsCoverage() && this.lastPartitions.size() != 16384) {
            this.internalShutdown();
            if (failedMasters.isEmpty()) {
                throw new RedisConnectionException("Not all slots covered! Only " + this.lastPartitions.size() + " slots are available. Set checkSlotsCoverage = false to avoid this check.", lastException);
            }
            throw new RedisConnectionException("Not all slots covered! Only " + this.lastPartitions.size() + " slots are available. Set checkSlotsCoverage = false to avoid this check. Failed masters according to cluster status: " + failedMasters, lastException);
        }
        if (this.cfg.getShardedSubscriptionMode() == ShardedSubscriptionMode.AUTO) {
            this.subscribeService.detectSharding();
        } else if (this.cfg.getShardedSubscriptionMode() == ShardedSubscriptionMode.ON) {
            this.subscribeService.setShardingSupported(true);
        }
        this.scheduleClusterChangeCheck(this.cfg);
    }

    @Override
    public Collection<MasterSlaveEntry> getEntrySet() {
        this.lazyConnect();
        return this.client2entry.values();
    }

    @Override
    public MasterSlaveEntry getEntry(RedisURI addr) {
        this.lazyConnect();
        for (MasterSlaveEntry entry : this.client2entry.values()) {
            if (addr.equals(entry.getClient().getAddr())) {
                return entry;
            }
            if (!entry.hasSlave(addr)) continue;
            return entry;
        }
        return null;
    }

    @Override
    public MasterSlaveEntry getEntry(RedisClient redisClient) {
        this.lazyConnect();
        MasterSlaveEntry entry = this.client2entry.get(redisClient);
        if (entry != null) {
            return entry;
        }
        for (MasterSlaveEntry mentry : this.client2entry.values()) {
            if (!mentry.hasSlave(redisClient)) continue;
            return mentry;
        }
        return null;
    }

    @Override
    public MasterSlaveEntry getEntry(InetSocketAddress address) {
        this.lazyConnect();
        for (MasterSlaveEntry entry : this.client2entry.values()) {
            InetSocketAddress addr = entry.getClient().getAddr();
            if (addr.getAddress().equals(address.getAddress()) && addr.getPort() == address.getPort()) {
                return entry;
            }
            if (!entry.hasSlave(address)) continue;
            return entry;
        }
        return null;
    }

    @Override
    protected CompletableFuture<RedisClient> changeMaster(int slot, RedisURI address) {
        MasterSlaveEntry entry = this.getEntry(slot);
        RedisClient oldClient = entry.getClient();
        CompletableFuture<RedisClient> future = super.changeMaster(slot, address);
        return future.thenApply(res -> {
            this.client2entry.remove(oldClient);
            this.client2entry.put(entry.getClient(), entry);
            return res;
        });
    }

    @Override
    public MasterSlaveEntry getEntry(int slot) {
        this.lazyConnect();
        return this.slot2entry.get(slot);
    }

    private void addEntry(Integer slot, MasterSlaveEntry entry) {
        MasterSlaveEntry oldEntry = this.slot2entry.getAndSet(slot, entry);
        if (oldEntry != entry) {
            entry.incReference();
            this.shutdownEntry(oldEntry);
        }
        this.client2entry.put(entry.getClient(), entry);
    }

    private void removeEntry(Integer slot) {
        MasterSlaveEntry entry = this.slot2entry.getAndSet(slot, null);
        this.shutdownEntry(entry);
    }

    private void shutdownEntry(MasterSlaveEntry entry) {
        if (entry != null && entry.decReference() == 0) {
            entry.getAllEntries().forEach(e -> {
                RedisURI uri = new RedisURI(e.getClient().getConfig().getAddress().getScheme(), e.getClient().getAddr().getAddress().getHostAddress(), e.getClient().getAddr().getPort());
                this.disconnectNode(uri);
                e.nodeDown();
            });
            entry.masterDown();
            entry.shutdownAsync();
            this.subscribeService.remove(entry);
            RedisURI uri = new RedisURI(entry.getClient().getConfig().getAddress().getScheme(), entry.getClient().getAddr().getAddress().getHostAddress(), entry.getClient().getAddr().getPort());
            this.disconnectNode(uri);
            this.client2entry.remove(entry.getClient());
            String slaves = entry.getAllEntries().stream().filter(e -> !e.getClient().getAddr().equals(entry.getClient().getAddr())).map(e -> e.getClient().toString()).collect(Collectors.joining(","));
            this.log.info("{} master and related slaves: {} removed", (Object)entry.getClient().getAddr(), (Object)slaves);
        }
    }

    @Override
    protected RedisClientConfig createRedisConfig(NodeType type, RedisURI address, int timeout, int commandTimeout, String sslHostname) {
        RedisClientConfig result = super.createRedisConfig(type, address, timeout, commandTimeout, sslHostname);
        result.setReadOnly(type == NodeType.SLAVE && this.config.getReadMode() != ReadMode.MASTER);
        return result;
    }

    private CompletionStage<Void> addMasterEntry(ClusterPartition partition, ClusterServersConfig cfg) {
        if (partition.isMasterFail()) {
            RedisException e = new RedisException("Failed to add master: " + partition.getMasterAddress() + " for slot ranges: " + partition.getSlotRanges() + ". Reason - server has FAIL flag");
            if (partition.getSlotsAmount() == 0) {
                e = new RedisException("Failed to add master: " + partition.getMasterAddress() + ". Reason - server has FAIL flag");
            }
            CompletableFuture<Void> result = new CompletableFuture<Void>();
            result.completeExceptionally(e);
            return result;
        }
        CompletionStage<RedisConnection> connectionFuture = this.connectToNode(cfg, partition.getMasterAddress(), this.configEndpointHostName);
        return connectionFuture.thenCompose(connection -> {
            MasterSlaveEntry entry;
            MasterSlaveServersConfig config = this.create(cfg);
            config.setMasterAddress(partition.getMasterAddress().toString());
            if (config.isSlaveNotUsed()) {
                entry = new SingleEntry(this, config);
            } else {
                Set<String> slaveAddresses = partition.getSlaveAddresses().stream().map(r -> r.toString()).collect(Collectors.toSet());
                config.setSlaveAddresses(slaveAddresses);
                entry = new MasterSlaveEntry(this, config);
            }
            CompletableFuture<RedisClient> f = entry.setupMasterEntry(new RedisURI(config.getMasterAddress()), this.configEndpointHostName);
            return f.thenCompose(masterClient -> {
                for (Integer slot : partition.getSlots()) {
                    this.addEntry(slot, entry);
                    this.lastPartitions.put(slot, partition);
                }
                if (partition.getSlotsAmount() > 0) {
                    this.lastUri2Partition.put(partition.getMasterAddress(), partition);
                }
                if (!config.isSlaveNotUsed()) {
                    CompletableFuture<Void> fs = entry.initSlaveBalancer(partition.getFailedSlaveAddresses(), r -> this.configEndpointHostName);
                    return fs.thenAccept(r -> {
                        if (!partition.getSlaveAddresses().isEmpty()) {
                            this.log.info("slaves: {} added for master: {} slot ranges: {}", partition.getSlaveAddresses(), partition.getMasterAddress(), partition.getSlotRanges());
                            if (!partition.getFailedSlaveAddresses().isEmpty()) {
                                this.log.warn("slaves: {} down for master: {} slot ranges: {}", partition.getFailedSlaveAddresses(), partition.getMasterAddress(), partition.getSlotRanges());
                            }
                        }
                        this.log.info("master: {} added for slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                    });
                }
                this.log.info("master: {} added for slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                return CompletableFuture.completedFuture(null);
            });
        });
    }

    private void scheduleClusterChangeCheck(ClusterServersConfig cfg) {
        this.monitorFuture = this.serviceManager.newTimeout(t -> {
            if (this.configEndpointHostName != null) {
                String address = cfg.getNodeAddresses().iterator().next();
                RedisURI uri = new RedisURI(address);
                CompletableFuture<List<RedisURI>> allNodes = this.serviceManager.resolveAll(uri);
                allNodes.whenComplete((nodes, ex) -> {
                    this.log.debug("{} resolved to {}", (Object)uri, nodes);
                    AtomicReference<Throwable> lastException = new AtomicReference<Throwable>((Throwable)ex);
                    if (ex != null) {
                        this.checkClusterState(cfg, Collections.emptyIterator(), lastException);
                        return;
                    }
                    Iterator<RedisURI> nodesIterator = nodes.iterator();
                    this.checkClusterState(cfg, nodesIterator, lastException);
                });
            } else {
                AtomicReference<Throwable> lastException = new AtomicReference<Throwable>();
                ArrayList<RedisURI> nodes2 = new ArrayList<RedisURI>();
                ArrayList<RedisURI> slaves = new ArrayList<RedisURI>();
                for (ClusterPartition partition : this.getLastPartitions()) {
                    if (!partition.isMasterFail()) {
                        nodes2.add(partition.getMasterAddress());
                    }
                    HashSet<RedisURI> partitionSlaves = new HashSet<RedisURI>(partition.getSlaveAddresses());
                    partitionSlaves.removeAll(partition.getFailedSlaveAddresses());
                    slaves.addAll(partitionSlaves);
                }
                Collections.shuffle(nodes2);
                Collections.shuffle(slaves);
                nodes2.addAll(slaves);
                Iterator<RedisURI> nodesIterator = nodes2.iterator();
                this.checkClusterState(cfg, nodesIterator, lastException);
            }
        }, cfg.getScanInterval(), TimeUnit.MILLISECONDS);
    }

    private void checkClusterState(ClusterServersConfig cfg, Iterator<RedisURI> iterator, AtomicReference<Throwable> lastException) {
        if (!iterator.hasNext()) {
            if (lastException.get() != null) {
                this.log.error("Can't update cluster state", lastException.get());
            }
            this.scheduleClusterChangeCheck(cfg);
            return;
        }
        if (this.serviceManager.isShuttingDown()) {
            return;
        }
        RedisURI uri = iterator.next();
        CompletionStage<RedisConnection> connectionFuture = this.connectToNode(cfg, uri, this.configEndpointHostName);
        connectionFuture.whenComplete((connection, e) -> {
            if (e != null) {
                lastException.set((Throwable)e);
                this.checkClusterState(cfg, iterator, lastException);
                return;
            }
            this.updateClusterState(cfg, (RedisConnection)connection, iterator, uri, lastException);
        });
    }

    private void updateClusterState(ClusterServersConfig cfg, RedisConnection connection, Iterator<RedisURI> iterator, RedisURI uri, AtomicReference<Throwable> lastException) {
        RFuture future = connection.async(this.clusterNodesCommand, new Object[0]);
        future.whenComplete((nodes, e) -> {
            if (e != null) {
                this.log.error("Unable to execute {}", (Object)this.clusterNodesCommand, e);
                lastException.set((Throwable)e);
                this.checkClusterState(cfg, iterator, lastException);
                return;
            }
            if (nodes.isEmpty()) {
                this.log.debug("cluster nodes state got from {}: doesn't contain any nodes", (Object)connection.getRedisClient().getAddr());
                this.checkClusterState(cfg, iterator, lastException);
                return;
            }
            this.lastClusterNode = uri;
            if (this.log.isDebugEnabled()) {
                StringBuilder nodesValue = new StringBuilder();
                for (ClusterNodeInfo clusterNodeInfo : nodes) {
                    nodesValue.append(clusterNodeInfo.getNodeInfo()).append("\n");
                }
                this.log.debug("cluster nodes state got from {}:\n{}", (Object)connection.getRedisClient().getAddr(), (Object)nodesValue);
            }
            CompletableFuture<Collection<ClusterPartition>> newPartitionsFuture = this.parsePartitions((List<ClusterNodeInfo>)nodes);
            ((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)newPartitionsFuture.whenComplete((r, ex) -> {
                if (ex != null) {
                    StringBuilder nodesValue = new StringBuilder();
                    for (ClusterNodeInfo clusterNodeInfo : nodes) {
                        nodesValue.append(clusterNodeInfo.getNodeInfo()).append("\n");
                    }
                    this.log.error("Unable to parse cluster nodes state got from: {}:\n{}", connection.getRedisClient().getAddr(), nodesValue, ex);
                    lastException.set((Throwable)ex);
                    this.checkClusterState(cfg, iterator, lastException);
                }
            })).thenCompose(newPartitions -> this.checkMasterNodesChange(cfg, (Collection<ClusterPartition>)newPartitions))).thenCompose(r -> newPartitionsFuture)).thenCompose(newPartitions -> this.checkSlaveNodesChange((Collection<ClusterPartition>)newPartitions))).thenCompose(r -> newPartitionsFuture)).thenApply(newPartitions -> {
                this.checkSlotsMigration((Collection<ClusterPartition>)newPartitions);
                this.checkSlotsChange((Collection<ClusterPartition>)newPartitions);
                this.scheduleClusterChangeCheck(cfg);
                return newPartitions;
            });
        });
    }

    private CompletableFuture<Void> checkSlaveNodesChange(Collection<ClusterPartition> newPartitions) {
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (ClusterPartition newPart : newPartitions) {
            ClusterPartition currentPart = this.lastUri2Partition.get(newPart.getMasterAddress());
            if (currentPart == null) continue;
            MasterSlaveEntry entry = this.getEntry(currentPart.getSlotRanges().iterator().next().getStartSlot());
            CompletableFuture<Set<RedisURI>> addedSlavesFuture = this.addRemoveSlaves(entry, currentPart, newPart);
            CompletionStage f = addedSlavesFuture.thenCompose(addedSlaves -> this.upDownSlaves(entry, currentPart, newPart, (Set<RedisURI>)addedSlaves));
            futures.add(f);
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).exceptionally(e -> {
            if (e != null) {
                this.log.error("Unable to add/remove slave nodes", (Throwable)e);
            }
            return null;
        });
    }

    private CompletableFuture<Void> upDownSlaves(MasterSlaveEntry entry, ClusterPartition currentPart, ClusterPartition newPart, Set<RedisURI> addedSlaves) {
        ArrayList futures = new ArrayList();
        List<RedisURI> nonFailedSlaves = currentPart.getFailedSlaveAddresses().stream().filter(uri -> !addedSlaves.contains(uri) && !newPart.getFailedSlaveAddresses().contains(uri)).collect(Collectors.toList());
        nonFailedSlaves.forEach(uri -> {
            if (entry.hasSlave((RedisURI)uri)) {
                CompletionStage<Boolean> f = entry.slaveUpNoMasterExclusionAsync((RedisURI)uri, ClientConnectionsEntry.FreezeReason.MANAGER);
                f = f.thenApply(v -> {
                    if (v.booleanValue()) {
                        this.log.info("slave: {} is up for slot ranges: {}", uri, (Object)currentPart.getSlotRanges());
                        currentPart.removeFailedSlaveAddress((RedisURI)uri);
                        entry.excludeMasterFromSlaves((RedisURI)uri);
                    }
                    return v;
                });
                futures.add(f);
            }
        });
        newPart.getFailedSlaveAddresses().stream().filter(uri -> !currentPart.getFailedSlaveAddresses().contains(uri)).forEach(uri -> {
            currentPart.addFailedSlaveAddress((RedisURI)uri);
            boolean slaveDown = entry.slaveDown((RedisURI)uri, ClientConnectionsEntry.FreezeReason.MANAGER);
            if (this.config.isSlaveNotUsed() || slaveDown) {
                this.disconnectNode((RedisURI)uri);
                this.log.warn("slave: {} has down for slot ranges: {}", uri, (Object)currentPart.getSlotRanges());
            }
        });
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    private CompletableFuture<Set<RedisURI>> addRemoveSlaves(MasterSlaveEntry entry, ClusterPartition currentPart, ClusterPartition newPart) {
        HashSet<RedisURI> removedSlaves = new HashSet<RedisURI>(currentPart.getSlaveAddresses());
        removedSlaves.removeAll(newPart.getSlaveAddresses());
        if (!removedSlaves.isEmpty()) {
            this.log.info("removed slaves detected for master {}. current slaves {} last slaves {}", currentPart.getMasterAddress(), currentPart.getSlaveAddresses(), newPart.getSlaveAddresses());
        }
        for (RedisURI uri2 : removedSlaves) {
            currentPart.removeSlaveAddress(uri2);
            boolean slaveDown = entry.slaveDown(uri2, ClientConnectionsEntry.FreezeReason.MANAGER);
            if (!this.config.isSlaveNotUsed() && !slaveDown) continue;
            this.disconnectNode(uri2);
            this.log.info("slave {} removed for master {} and slot ranges: {}", currentPart.getMasterAddress(), uri2, currentPart.getSlotRanges());
        }
        Set addedSlaves = newPart.getSlaveAddresses().stream().filter(uri -> !currentPart.getSlaveAddresses().contains(uri) && !newPart.getFailedSlaveAddresses().contains(uri)).collect(Collectors.toSet());
        if (!addedSlaves.isEmpty()) {
            this.log.info("added slaves detected for master {}. current slaves {} last slaves {} last failed slaves {}", currentPart.getMasterAddress(), currentPart.getSlaveAddresses(), newPart.getSlaveAddresses(), newPart.getFailedSlaveAddresses());
        }
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (RedisURI uri3 : addedSlaves) {
            CompletionStage<Object> slaveUpFuture;
            ClientConnectionsEntry slaveEntry = entry.getEntry(uri3);
            if (slaveEntry != null) {
                slaveUpFuture = entry.slaveUpNoMasterExclusionAsync(uri3, ClientConnectionsEntry.FreezeReason.MANAGER);
                slaveUpFuture = slaveUpFuture.thenApply(v -> {
                    if (v.booleanValue()) {
                        currentPart.addSlaveAddress(uri3);
                        this.log.info("slave: {} unfreezed for master {} and slot ranges: {}", currentPart.getMasterAddress(), uri3, currentPart.getSlotRanges());
                        entry.excludeMasterFromSlaves(uri3);
                    }
                    return v;
                });
                futures.add(slaveUpFuture);
                continue;
            }
            slaveUpFuture = entry.addSlave(uri3, false, this.configEndpointHostName);
            CompletionStage f = slaveUpFuture.thenAccept(res -> {
                currentPart.addSlaveAddress(uri3);
                this.log.info("slave: {} added for master {} and slot ranges: {}", currentPart.getMasterAddress(), uri3, currentPart.getSlotRanges());
                entry.excludeMasterFromSlaves(uri3);
            });
            futures.add(f);
        }
        CompletableFuture<Void> f = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
        return f.thenApply(r -> addedSlaves);
    }

    private ClusterPartition find(Collection<ClusterPartition> partitions, Integer slot) {
        return partitions.stream().filter(p -> p.hasSlot(slot)).findFirst().orElseThrow(() -> new IllegalStateException("Unable to find partition with slot " + slot));
    }

    private CompletableFuture<Void> checkMasterNodesChange(ClusterServersConfig cfg, Collection<ClusterPartition> newPartitions) {
        HashMap<RedisURI, ClusterPartition> addedPartitions = new HashMap<RedisURI, ClusterPartition>();
        HashSet<RedisURI> mastersElected = new HashSet<RedisURI>();
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (ClusterPartition newPart : newPartitions) {
            boolean masterFound;
            if (newPart.getSlotsAmount() == 0) continue;
            ClusterPartition currentPart = this.lastUri2Partition.get(newPart.getMasterAddress());
            boolean bl = masterFound = currentPart != null;
            if (masterFound && newPart.isMasterFail()) {
                for (Integer slot : currentPart.getSlots()) {
                    ClusterPartition newMasterPart = this.find(newPartitions, slot);
                    if (Objects.equals(newMasterPart.getMasterAddress(), currentPart.getMasterAddress())) continue;
                    RedisURI newUri = newMasterPart.getMasterAddress();
                    RedisURI oldUri = currentPart.getMasterAddress();
                    mastersElected.add(newUri);
                    CompletableFuture<RedisClient> future = this.changeMaster(slot, newUri);
                    currentPart.setMasterAddress(newUri);
                    CompletionStage f = future.whenComplete((res, e) -> {
                        if (e != null) {
                            currentPart.setMasterAddress(oldUri);
                        } else {
                            this.disconnectNode(oldUri);
                        }
                    });
                    futures.add(f);
                }
            }
            if (masterFound || newPart.isMasterFail()) continue;
            addedPartitions.put(newPart.getMasterAddress(), newPart);
        }
        addedPartitions.keySet().removeAll(mastersElected);
        for (ClusterPartition newPart : addedPartitions.values()) {
            CompletionStage<Void> future = this.addMasterEntry(newPart, cfg);
            futures.add(future.toCompletableFuture());
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).exceptionally(e -> {
            if (e != null) {
                this.log.error("Unable to add/change master node", (Throwable)e);
            }
            return null;
        });
    }

    private void checkSlotsChange(Collection<ClusterPartition> newPartitions) {
        int newSlotsAmount = newPartitions.stream().mapToInt(ClusterPartition::getSlotsAmount).sum();
        if (newSlotsAmount == this.lastPartitions.size() && this.lastPartitions.size() == 16384) {
            return;
        }
        Set removedSlots = this.lastPartitions.keySet().stream().filter(s -> newPartitions.stream().noneMatch(p -> p.hasSlot((int)s))).collect(Collectors.toSet());
        for (Integer removedSlot : removedSlots) {
            ClusterPartition p = this.lastPartitions.remove(removedSlot);
            if (p == null) continue;
            this.lastUri2Partition.remove(p.getMasterAddress());
        }
        if (!removedSlots.isEmpty()) {
            this.log.info("{} slots found to remove", (Object)removedSlots.size());
        }
        for (Integer slot : removedSlots) {
            this.removeEntry(slot);
        }
        Integer addedSlots = 0;
        for (ClusterPartition clusterPartition : newPartitions) {
            MasterSlaveEntry entry = this.getEntry(clusterPartition.getMasterAddress());
            for (Integer slot : clusterPartition.getSlots()) {
                if (this.lastPartitions.containsKey(slot) || entry == null) continue;
                this.addEntry(slot, entry);
                this.lastPartitions.put(slot, clusterPartition);
                this.lastUri2Partition.put(clusterPartition.getMasterAddress(), clusterPartition);
                Integer n = addedSlots;
                addedSlots = addedSlots + 1;
            }
        }
        if (addedSlots > 0) {
            this.log.info("{} slots found to add", (Object)addedSlots);
        }
    }

    private void checkSlotsMigration(Collection<ClusterPartition> newPartitions) {
        Collection<ClusterPartition> clusterLastPartitions = this.getLastPartitions();
        Map<String, MasterSlaveEntry> nodeEntries = clusterLastPartitions.stream().collect(Collectors.toMap(p -> p.getNodeId(), p -> this.getEntry(p.getSlotRanges().iterator().next().getStartSlot())));
        HashSet changedSlots = new HashSet();
        block0: for (ClusterPartition currentPartition : clusterLastPartitions) {
            String nodeId = currentPartition.getNodeId();
            for (ClusterPartition newPartition : newPartitions) {
                if (!Objects.equals(nodeId, newPartition.getNodeId()) || newPartition.getSlotRanges().equals(currentPartition.getSlotRanges())) continue;
                MasterSlaveEntry entry = nodeEntries.get(nodeId);
                BitSet addedSlots = newPartition.copySlots();
                addedSlots.andNot(currentPartition.slots());
                addedSlots.stream().forEach(slot -> {
                    this.addEntry(slot, entry);
                    this.lastPartitions.put(slot, currentPartition);
                    changedSlots.add(slot);
                });
                if (!addedSlots.isEmpty()) {
                    this.lastUri2Partition.put(currentPartition.getMasterAddress(), currentPartition);
                    this.log.info("{} slots added to {}", (Object)addedSlots.cardinality(), (Object)currentPartition.getMasterAddress());
                }
                BitSet removedSlots = currentPartition.copySlots();
                removedSlots.andNot(newPartition.slots());
                removedSlots.stream().forEach(slot -> {
                    if (this.lastPartitions.remove(slot, currentPartition)) {
                        this.lastUri2Partition.remove(currentPartition.getMasterAddress());
                        this.removeEntry(slot);
                        changedSlots.add(slot);
                    }
                });
                if (!removedSlots.isEmpty()) {
                    this.log.info("{} slots removed from {}", (Object)removedSlots.cardinality(), (Object)currentPartition.getMasterAddress());
                }
                if (addedSlots.isEmpty() && removedSlots.isEmpty()) continue block0;
                currentPartition.updateSlotRanges(newPartition.getSlotRanges(), newPartition.slots());
                continue block0;
            }
        }
        changedSlots.forEach(this.subscribeService::reattachPubSub);
    }

    private int indexOf(byte[] array, byte element) {
        for (int i = 0; i < array.length; ++i) {
            if (array[i] != element) continue;
            return i;
        }
        return -1;
    }

    @Override
    public int calcSlot(byte[] key) {
        int end;
        if (key == null) {
            return 0;
        }
        int start = this.indexOf(key, (byte)123);
        if (start != -1 && (end = this.indexOf(key, (byte)125)) != -1 && start + 1 < end) {
            key = Arrays.copyOfRange(key, start + 1, end);
        }
        int result = CRC16.crc16(key) % 16384;
        return result;
    }

    @Override
    public int calcSlot(ByteBuf key) {
        int end;
        if (key == null) {
            return 0;
        }
        int start = key.indexOf(key.readerIndex(), key.readerIndex() + key.readableBytes(), (byte)123);
        if (start != -1 && (end = key.indexOf(start + 1, key.readerIndex() + key.readableBytes(), (byte)125)) != -1 && start + 1 < end) {
            key = key.slice(start + 1, end - start - 1);
        }
        int result = CRC16.crc16(key) % 16384;
        this.log.debug("slot {} for {}", (Object)result, (Object)key);
        return result;
    }

    @Override
    public int calcSlot(String key) {
        int end;
        if (key == null) {
            return 0;
        }
        int start = key.indexOf(123);
        if (start != -1 && (end = key.indexOf(125)) != -1 && start + 1 < end) {
            key = key.substring(start + 1, end);
        }
        int result = CRC16.crc16(key.getBytes()) % 16384;
        this.log.debug("slot {} for {}", (Object)result, (Object)key);
        return result;
    }

    private CompletableFuture<Collection<ClusterPartition>> parsePartitions(List<ClusterNodeInfo> nodes) {
        ConcurrentHashMap partitions = new ConcurrentHashMap();
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (ClusterNodeInfo clusterNodeInfo : nodes) {
            String masterId;
            if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.NOADDR) || clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.HANDSHAKE) || clusterNodeInfo.getAddress() == null || clusterNodeInfo.getSlotRanges().isEmpty() && clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.MASTER) || (masterId = clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.SLAVE) ? clusterNodeInfo.getSlaveOf() : clusterNodeInfo.getNodeId()) == null) continue;
            CompletableFuture<List<RedisURI>> ipsFuture = this.serviceManager.resolveAll(clusterNodeInfo.getAddress());
            CompletionStage f = ((CompletableFuture)ipsFuture.thenAccept(addresses -> {
                int index = 0;
                if (addresses.size() > 1) {
                    addresses.sort(Comparator.comparing(RedisURI::getHost));
                    SplittableRandom r = new SplittableRandom(this.seed);
                    index = r.nextInt(addresses.size());
                }
                RedisURI address = (RedisURI)addresses.get(index);
                if (addresses.size() == 1) {
                    this.log.debug("{} resolved to {}", (Object)clusterNodeInfo.getAddress(), (Object)address);
                } else {
                    this.log.debug("{} resolved to {} and {} selected", clusterNodeInfo.getAddress(), addresses, address);
                }
                if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.SLAVE)) {
                    ClusterPartition masterPartition = partitions.computeIfAbsent(masterId, k -> new ClusterPartition(masterId));
                    ClusterPartition slavePartition = partitions.computeIfAbsent(clusterNodeInfo.getNodeId(), k -> new ClusterPartition(clusterNodeInfo.getNodeId()));
                    slavePartition.setType(ClusterPartition.Type.SLAVE);
                    slavePartition.setParent(masterPartition);
                    masterPartition.addSlaveAddress(address);
                    if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.FAIL)) {
                        masterPartition.addFailedSlaveAddress(address);
                    }
                } else if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.MASTER)) {
                    ClusterPartition masterPartition = partitions.computeIfAbsent(masterId, k -> new ClusterPartition(masterId));
                    masterPartition.setSlotRanges(clusterNodeInfo.getSlotRanges());
                    masterPartition.setMasterAddress(address);
                    masterPartition.setType(ClusterPartition.Type.MASTER);
                    if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.FAIL)) {
                        masterPartition.setMasterFail(true);
                    }
                }
            })).exceptionally(ex -> {
                if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.FAIL) || clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.EVENTUAL_FAIL)) {
                    return null;
                }
                this.log.error(ex.getMessage(), (Throwable)ex);
                return null;
            });
            futures.add(f);
        }
        CompletableFuture<Void> future = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
        return future.thenApply(r -> {
            this.addCascadeSlaves(partitions.values());
            List ps = partitions.values().stream().filter(cp -> cp.getType() == ClusterPartition.Type.MASTER && cp.getMasterAddress() != null && (!cp.slots().isEmpty() && partitions.size() == 1 || partitions.size() > 1)).collect(Collectors.toList());
            return ps;
        });
    }

    private void addCascadeSlaves(Collection<ClusterPartition> partitions) {
        Iterator<ClusterPartition> iter = partitions.iterator();
        while (iter.hasNext()) {
            ClusterPartition cp = iter.next();
            if (cp.getType() != ClusterPartition.Type.SLAVE) continue;
            if (cp.getParent() != null && cp.getParent().getType() == ClusterPartition.Type.MASTER) {
                ClusterPartition parent = cp.getParent();
                for (RedisURI addr : cp.getSlaveAddresses()) {
                    parent.addSlaveAddress(addr);
                }
                for (RedisURI addr : cp.getFailedSlaveAddresses()) {
                    parent.addFailedSlaveAddress(addr);
                }
            }
            iter.remove();
        }
    }

    @Override
    public void shutdown(long quietPeriod, long timeout, TimeUnit unit) {
        if (this.monitorFuture != null) {
            this.monitorFuture.cancel();
        }
        this.closeNodeConnections();
        super.shutdown(quietPeriod, timeout, unit);
    }

    private Collection<ClusterPartition> getLastPartitions() {
        return this.lastUri2Partition.values();
    }

    public int getSlot(MasterSlaveEntry entry) {
        return this.lastPartitions.entrySet().stream().filter(e -> ((ClusterPartition)e.getValue()).getMasterAddress().equals(entry.getClient().getConfig().getAddress())).findAny().map(m -> (Integer)m.getKey()).orElse(-1);
    }

    @Override
    public RedisURI getLastClusterNode() {
        return this.lastClusterNode;
    }

    @Override
    public boolean isClusterMode() {
        return true;
    }
}

