/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection.pool;

import java.net.InetSocketAddress;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import org.redisson.api.NodeType;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.protocol.RedisCommand;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class ConnectionPool<T extends RedisConnection> {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    protected final Queue<ClientConnectionsEntry> entries = new ConcurrentLinkedQueue<ClientConnectionsEntry>();
    final ConnectionManager connectionManager;
    final MasterSlaveServersConfig config;
    final MasterSlaveEntry masterSlaveEntry;

    ConnectionPool(MasterSlaveServersConfig config, ConnectionManager connectionManager, MasterSlaveEntry masterSlaveEntry) {
        this.config = config;
        this.masterSlaveEntry = masterSlaveEntry;
        this.connectionManager = connectionManager;
    }

    public void addEntry(ClientConnectionsEntry entry) {
        this.entries.add(entry);
    }

    public CompletableFuture<Void> initConnections(ClientConnectionsEntry entry) {
        int minimumIdleSize = this.getMinimumIdleSize(entry);
        if (minimumIdleSize == 0) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<Void> initPromise = new CompletableFuture<Void>();
        AtomicInteger initializedConnections = new AtomicInteger(minimumIdleSize);
        this.createConnection(entry, initPromise, minimumIdleSize, initializedConnections);
        return initPromise;
    }

    private void createConnection(ClientConnectionsEntry entry, CompletableFuture<Void> initPromise, int minimumIdleSize, AtomicInteger initializedConnections) {
        CompletableFuture<Void> f = this.acquireConnection(entry, null);
        f.thenAccept(r -> {
            CompletableFuture promise = new CompletableFuture();
            this.createConnection(entry, promise);
            promise.whenComplete((conn, e) -> {
                if (e == null) {
                    if (this.changeUsage()) {
                        conn.decUsage();
                    }
                    if (!initPromise.isDone()) {
                        entry.addConnection((RedisConnection)conn);
                    } else {
                        conn.closeAsync();
                    }
                }
                this.releaseConnection(entry);
                if (e != null) {
                    void var8_15;
                    if (initPromise.isDone()) {
                        return;
                    }
                    for (RedisConnection redisConnection : entry.getAllConnections()) {
                        if (redisConnection.isClosed()) continue;
                        redisConnection.closeAsync();
                    }
                    entry.getAllConnections().clear();
                    for (RedisConnection redisConnection : entry.getAllSubscribeConnections()) {
                        if (redisConnection.isClosed()) continue;
                        redisConnection.closeAsync();
                    }
                    entry.getAllSubscribeConnections().clear();
                    int totalInitializedConnections = minimumIdleSize - initializedConnections.get();
                    if (totalInitializedConnections == 0) {
                        String string = "Unable to connect to Redis server: " + entry.getClient().getAddr();
                    } else {
                        String string = "Unable to init enough connections amount! Only " + totalInitializedConnections + " of " + minimumIdleSize + " were initialized. Redis server: " + entry.getClient().getAddr();
                    }
                    RedisConnectionException cause = new RedisConnectionException((String)var8_15, (Throwable)e);
                    initPromise.completeExceptionally(cause);
                    return;
                }
                int value = initializedConnections.decrementAndGet();
                if (value == 0) {
                    if (initPromise.complete(null)) {
                        this.log.info("{} connections initialized for {}", (Object)minimumIdleSize, (Object)entry.getClient().getAddr());
                    }
                } else if (value > 0 && !initPromise.isDone()) {
                    this.createConnection(entry, initPromise, minimumIdleSize, initializedConnections);
                }
            });
        });
    }

    protected CompletableFuture<Void> acquireConnection(ClientConnectionsEntry entry, RedisCommand<?> command) {
        return entry.acquireConnection(command);
    }

    protected abstract int getMinimumIdleSize(ClientConnectionsEntry var1);

    public CompletableFuture<T> get(RedisCommand<?> command) {
        LinkedList<ClientConnectionsEntry> entriesCopy = new LinkedList<ClientConnectionsEntry>(this.entries);
        Iterator iterator = entriesCopy.iterator();
        while (iterator.hasNext()) {
            ClientConnectionsEntry entry = (ClientConnectionsEntry)iterator.next();
            if ((!entry.isFreezed() || entry.isMasterForRead()) && this.isHealthy(entry)) continue;
            iterator.remove();
        }
        if (!entriesCopy.isEmpty()) {
            ClientConnectionsEntry entry = this.config.getLoadBalancer().getEntry(entriesCopy, command);
            return this.acquireConnection(command, entry);
        }
        LinkedList<InetSocketAddress> failed = new LinkedList<InetSocketAddress>();
        LinkedList<InetSocketAddress> freezed = new LinkedList<InetSocketAddress>();
        for (ClientConnectionsEntry entry : this.entries) {
            if (entry.getClient().getConfig().getFailedNodeDetector().isNodeFailed()) {
                failed.add(entry.getClient().getAddr());
                continue;
            }
            if (!entry.isFreezed()) continue;
            freezed.add(entry.getClient().getAddr());
        }
        StringBuilder errorMsg = new StringBuilder(this.getClass().getSimpleName() + " no available Redis entries. Master entry host: " + this.masterSlaveEntry.getClient().getAddr());
        if (!freezed.isEmpty()) {
            errorMsg.append(" Disconnected hosts: ").append(freezed);
        }
        if (!failed.isEmpty()) {
            errorMsg.append(" Hosts disconnected due to errors during `failedSlaveCheckInterval`: ").append(failed);
        }
        RedisConnectionException exception = new RedisConnectionException(errorMsg.toString());
        CompletableFuture result = new CompletableFuture();
        result.completeExceptionally(exception);
        return result;
    }

    public CompletableFuture<T> get(RedisCommand<?> command, ClientConnectionsEntry entry) {
        return this.acquireConnection(command, entry);
    }

    protected final CompletableFuture<T> acquireConnection(RedisCommand<?> command, ClientConnectionsEntry entry) {
        CompletableFuture result = new CompletableFuture();
        CompletableFuture<Void> f = this.acquireConnection(entry, command);
        f.thenAccept(r -> this.connectTo(entry, result, command));
        result.whenComplete((r, e) -> {
            if (e != null) {
                f.completeExceptionally((Throwable)e);
            }
        });
        return result;
    }

    private boolean isHealthy(ClientConnectionsEntry entry) {
        return entry.getNodeType() != NodeType.SLAVE || !entry.getClient().getConfig().getFailedNodeDetector().isNodeFailed();
    }

    protected T poll(ClientConnectionsEntry entry, RedisCommand<?> command) {
        return (T)entry.pollConnection(command);
    }

    protected CompletionStage<T> connect(ClientConnectionsEntry entry) {
        return entry.connect();
    }

    private void connectTo(ClientConnectionsEntry entry, CompletableFuture<T> promise, RedisCommand<?> command) {
        if (promise.isDone()) {
            this.connectionManager.getServiceManager().getGroup().submit(() -> this.releaseConnection(entry));
            return;
        }
        T conn = this.poll(entry, command);
        if (conn != null) {
            this.connectedSuccessful(entry, promise, conn);
            return;
        }
        this.createConnection(entry, promise);
    }

    private void createConnection(ClientConnectionsEntry entry, CompletableFuture<T> promise) {
        CompletionStage<RedisConnection> connFuture = this.connect(entry);
        connFuture.whenComplete((conn, e) -> {
            if (e != null) {
                this.promiseFailure(entry, promise, (Throwable)e);
                return;
            }
            if (this.changeUsage()) {
                promise.thenApply(c -> c.incUsage());
            }
            this.connectedSuccessful(entry, promise, conn);
        });
    }

    protected boolean changeUsage() {
        return true;
    }

    private void connectedSuccessful(ClientConnectionsEntry entry, CompletableFuture<T> promise, T conn) {
        if (entry.getNodeType() == NodeType.SLAVE) {
            entry.getClient().getConfig().getFailedNodeDetector().onConnectSuccessful();
        }
        if (!promise.complete(conn)) {
            this.releaseConnection(entry, conn);
            this.releaseConnection(entry);
        }
    }

    private void promiseFailure(ClientConnectionsEntry entry, CompletableFuture<T> promise, Throwable cause) {
        if (entry.getNodeType() == NodeType.SLAVE) {
            entry.getClient().getConfig().getFailedNodeDetector().onConnectFailed();
            if (entry.getClient().getConfig().getFailedNodeDetector().isNodeFailed()) {
                this.masterSlaveEntry.shutdownAndReconnectAsync(entry.getClient(), cause);
            }
        }
        this.releaseConnection(entry);
        promise.completeExceptionally(cause);
    }

    public void returnConnection(ClientConnectionsEntry entry, T connection) {
        if (entry == null) {
            ((RedisConnection)connection).closeAsync();
            return;
        }
        if (entry.isFreezed() && entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.SYSTEM) {
            ((RedisConnection)connection).closeAsync();
            entry.getAllConnections().remove(connection);
        } else {
            this.releaseConnection(entry, connection);
        }
        this.releaseConnection(entry);
    }

    protected void releaseConnection(ClientConnectionsEntry entry) {
        entry.releaseConnection();
    }

    protected void releaseConnection(ClientConnectionsEntry entry, T conn) {
        entry.releaseConnection((RedisConnection)conn);
    }
}

