/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection.pool;

import io.netty.util.Timeout;
import io.netty.util.TimerTask;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import java.net.InetSocketAddress;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.redisson.api.NodeType;
import org.redisson.api.RFuture;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.protocol.RedisCommand;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.redisson.misc.RPromise;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class ConnectionPool<T extends RedisConnection> {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    protected final List<ClientConnectionsEntry> entries = new CopyOnWriteArrayList<ClientConnectionsEntry>();
    final ConnectionManager connectionManager;
    final MasterSlaveServersConfig config;
    final MasterSlaveEntry masterSlaveEntry;

    public ConnectionPool(MasterSlaveServersConfig config, ConnectionManager connectionManager, MasterSlaveEntry masterSlaveEntry) {
        this.config = config;
        this.masterSlaveEntry = masterSlaveEntry;
        this.connectionManager = connectionManager;
    }

    public RFuture<Void> add(final ClientConnectionsEntry entry) {
        RPromise<Void> promise = this.connectionManager.newPromise();
        promise.addListener(new FutureListener<Void>(){

            @Override
            public void operationComplete(Future<Void> future) throws Exception {
                ConnectionPool.this.entries.add(entry);
            }
        });
        this.initConnections(entry, promise, true);
        return promise;
    }

    private void initConnections(ClientConnectionsEntry entry, RPromise<Void> initPromise, boolean checkFreezed) {
        int minimumIdleSize = this.getMinimumIdleSize(entry);
        if (minimumIdleSize == 0 || checkFreezed && entry.isFreezed()) {
            initPromise.trySuccess(null);
            return;
        }
        AtomicInteger initializedConnections = new AtomicInteger(minimumIdleSize);
        int startAmount = Math.min(50, minimumIdleSize);
        AtomicInteger requests = new AtomicInteger(startAmount);
        for (int i = 0; i < startAmount; ++i) {
            this.createConnection(checkFreezed, requests, entry, initPromise, minimumIdleSize, initializedConnections);
        }
    }

    private void createConnection(final boolean checkFreezed, final AtomicInteger requests, final ClientConnectionsEntry entry, final RPromise<Void> initPromise, final int minimumIdleSize, final AtomicInteger initializedConnections) {
        if (checkFreezed && entry.isFreezed() || !this.tryAcquireConnection(entry)) {
            int totalInitializedConnections = minimumIdleSize - initializedConnections.get();
            RedisConnectionException cause = new RedisConnectionException("Unable to init enough connections amount! Only " + totalInitializedConnections + " from " + minimumIdleSize + " were initialized. Server: " + entry.getClient().getAddr());
            initPromise.tryFailure(cause);
            return;
        }
        this.acquireConnection(entry, new Runnable(){

            @Override
            public void run() {
                RPromise promise = ConnectionPool.this.connectionManager.newPromise();
                ConnectionPool.this.createConnection(entry, promise);
                promise.addListener(new FutureListener<T>(){

                    @Override
                    public void operationComplete(Future<T> future) throws Exception {
                        if (future.isSuccess()) {
                            RedisConnection conn = (RedisConnection)future.getNow();
                            ConnectionPool.this.releaseConnection(entry, conn);
                        }
                        ConnectionPool.this.releaseConnection(entry);
                        if (!future.isSuccess()) {
                            int totalInitializedConnections = minimumIdleSize - initializedConnections.get();
                            String errorMsg = totalInitializedConnections == 0 ? "Unable to connect to Redis server: " + entry.getClient().getAddr() : "Unable to init enough connections amount! Only " + totalInitializedConnections + " from " + minimumIdleSize + " were initialized. Redis server: " + entry.getClient().getAddr();
                            RedisConnectionException cause = new RedisConnectionException(errorMsg, future.cause());
                            initPromise.tryFailure(cause);
                            return;
                        }
                        int value = initializedConnections.decrementAndGet();
                        if (value == 0) {
                            ConnectionPool.this.log.info("{} connections initialized for {}", (Object)minimumIdleSize, (Object)entry.getClient().getAddr());
                            if (!initPromise.trySuccess(null)) {
                                throw new IllegalStateException();
                            }
                        } else if (value > 0 && !initPromise.isDone() && requests.incrementAndGet() <= minimumIdleSize) {
                            ConnectionPool.this.createConnection(checkFreezed, requests, entry, initPromise, minimumIdleSize, initializedConnections);
                        }
                    }
                });
            }
        });
    }

    protected void acquireConnection(ClientConnectionsEntry entry, Runnable runnable) {
        entry.acquireConnection(runnable);
    }

    protected abstract int getMinimumIdleSize(ClientConnectionsEntry var1);

    protected ClientConnectionsEntry getEntry() {
        return this.config.getLoadBalancer().getEntry(this.entries);
    }

    public RFuture<T> get(RedisCommand<?> command) {
        for (int j = this.entries.size() - 1; j >= 0; --j) {
            ClientConnectionsEntry entry = this.getEntry();
            if (entry.isFreezed() || !this.tryAcquireConnection(entry)) continue;
            return this.acquireConnection(command, entry);
        }
        LinkedList<InetSocketAddress> failedAttempts = new LinkedList<InetSocketAddress>();
        LinkedList<InetSocketAddress> freezed = new LinkedList<InetSocketAddress>();
        for (ClientConnectionsEntry entry : this.entries) {
            if (entry.isFreezed()) {
                freezed.add(entry.getClient().getAddr());
                continue;
            }
            failedAttempts.add(entry.getClient().getAddr());
        }
        StringBuilder errorMsg = new StringBuilder(this.getClass().getSimpleName() + " no available Redis entries. ");
        if (!freezed.isEmpty()) {
            errorMsg.append(" Disconnected hosts: " + freezed);
        }
        if (!failedAttempts.isEmpty()) {
            errorMsg.append(" Hosts disconnected due to `failedAttempts` limit reached: " + failedAttempts);
        }
        RedisConnectionException exception = new RedisConnectionException(errorMsg.toString());
        return this.connectionManager.newFailedFuture(exception);
    }

    public RFuture<T> get(RedisCommand<?> command, ClientConnectionsEntry entry) {
        if ((entry.getNodeType() == NodeType.MASTER && entry.getFreezeReason() == ClientConnectionsEntry.FreezeReason.SYSTEM || !entry.isFreezed()) && this.tryAcquireConnection(entry)) {
            return this.acquireConnection(command, entry);
        }
        RedisConnectionException exception = new RedisConnectionException("Can't aquire connection to " + entry.getClient().getAddr());
        return this.connectionManager.newFailedFuture(exception);
    }

    private RFuture<T> acquireConnection(RedisCommand<?> command, final ClientConnectionsEntry entry) {
        final RPromise result = this.connectionManager.newPromise();
        AcquireCallback callback = new AcquireCallback<T>(){

            @Override
            public void run() {
                result.removeListener(this);
                ConnectionPool.this.connectTo(entry, result);
            }

            @Override
            public void operationComplete(Future<T> future) throws Exception {
                entry.removeConnection(this);
            }
        };
        result.addListener(callback);
        this.acquireConnection(entry, callback);
        return result;
    }

    protected boolean tryAcquireConnection(ClientConnectionsEntry entry) {
        return entry.getFailedAttempts() < this.config.getFailedAttempts();
    }

    protected T poll(ClientConnectionsEntry entry) {
        return (T)entry.pollConnection();
    }

    protected RFuture<T> connect(ClientConnectionsEntry entry) {
        return entry.connect();
    }

    private void connectTo(ClientConnectionsEntry entry, RPromise<T> promise) {
        if (promise.isDone()) {
            this.releaseConnection(entry);
            return;
        }
        T conn = this.poll(entry);
        if (conn != null) {
            if (!((RedisConnection)conn).isActive()) {
                this.promiseFailure(entry, promise, conn);
                return;
            }
            this.connectedSuccessful(entry, promise, conn);
            return;
        }
        this.createConnection(entry, promise);
    }

    private void createConnection(final ClientConnectionsEntry entry, final RPromise<T> promise) {
        RFuture<T> connFuture = this.connect(entry);
        connFuture.addListener(new FutureListener<T>(){

            @Override
            public void operationComplete(Future<T> future) throws Exception {
                if (!future.isSuccess()) {
                    ConnectionPool.this.promiseFailure(entry, (RPromise)promise, (Object)future.cause());
                    return;
                }
                RedisConnection conn = (RedisConnection)future.getNow();
                if (!conn.isActive()) {
                    ConnectionPool.this.promiseFailure(entry, promise, conn);
                    return;
                }
                ConnectionPool.this.connectedSuccessful(entry, promise, conn);
            }
        });
    }

    private void connectedSuccessful(ClientConnectionsEntry entry, RPromise<T> promise, T conn) {
        entry.resetFailedAttempts();
        if (!promise.trySuccess(conn)) {
            this.releaseConnection(entry, conn);
            this.releaseConnection(entry);
        }
    }

    private void promiseFailure(ClientConnectionsEntry entry, RPromise<T> promise, Throwable cause) {
        if (entry.incFailedAttempts() == this.config.getFailedAttempts()) {
            this.checkForReconnect(entry);
        }
        this.releaseConnection(entry);
        promise.tryFailure(cause);
    }

    private void promiseFailure(ClientConnectionsEntry entry, RPromise<T> promise, T conn) {
        int attempts = entry.incFailedAttempts();
        if (attempts == this.config.getFailedAttempts()) {
            ((RedisConnection)conn).closeAsync();
            this.checkForReconnect(entry);
        } else if (attempts < this.config.getFailedAttempts()) {
            this.releaseConnection(entry, conn);
        } else {
            ((RedisConnection)conn).closeAsync();
        }
        this.releaseConnection(entry);
        RedisConnectionException cause = new RedisConnectionException(conn + " is not active!");
        promise.tryFailure(cause);
    }

    private void checkForReconnect(ClientConnectionsEntry entry) {
        if (entry.getNodeType() == NodeType.SLAVE) {
            this.masterSlaveEntry.slaveDown(entry.getClient().getAddr().getHostName(), entry.getClient().getAddr().getPort(), ClientConnectionsEntry.FreezeReason.RECONNECT);
            this.log.warn("slave {} disconnected due to failedAttempts={} limit reached", (Object)entry.getClient().getAddr(), (Object)this.config.getFailedAttempts());
            this.scheduleCheck(entry);
        } else if (entry.freezeMaster(ClientConnectionsEntry.FreezeReason.RECONNECT)) {
            this.log.warn("host {} disconnected due to failedAttempts={} limit reached", (Object)entry.getClient().getAddr(), (Object)this.config.getFailedAttempts());
            this.scheduleCheck(entry);
        }
    }

    private void scheduleCheck(final ClientConnectionsEntry entry) {
        this.connectionManager.getConnectionEventsHub().fireDisconnect(entry.getClient().getAddr());
        this.connectionManager.newTimeout(new TimerTask(){

            @Override
            public void run(Timeout timeout) throws Exception {
                if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT || !entry.isFreezed() || ConnectionPool.this.connectionManager.isShuttingDown()) {
                    return;
                }
                RFuture<RedisConnection> connectionFuture = entry.getClient().connectAsync();
                connectionFuture.addListener(new FutureListener<RedisConnection>(){

                    @Override
                    public void operationComplete(Future<RedisConnection> future) throws Exception {
                        if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT || !entry.isFreezed()) {
                            return;
                        }
                        if (!future.isSuccess()) {
                            ConnectionPool.this.scheduleCheck(entry);
                            return;
                        }
                        final RedisConnection c = future.getNow();
                        if (!c.isActive()) {
                            c.closeAsync();
                            ConnectionPool.this.scheduleCheck(entry);
                            return;
                        }
                        final FutureListener<String> pingListener = new FutureListener<String>(){

                            /*
                             * WARNING - Removed try catching itself - possible behaviour change.
                             */
                            @Override
                            public void operationComplete(Future<String> future) throws Exception {
                                try {
                                    if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT || !entry.isFreezed()) {
                                        return;
                                    }
                                    if (future.isSuccess() && "PONG".equals(future.getNow())) {
                                        entry.resetFailedAttempts();
                                        RPromise promise = ConnectionPool.this.connectionManager.newPromise();
                                        promise.addListener(new FutureListener<Void>(){

                                            /*
                                             * WARNING - Removed try catching itself - possible behaviour change.
                                             */
                                            @Override
                                            public void operationComplete(Future<Void> future) throws Exception {
                                                if (entry.getNodeType() == NodeType.SLAVE) {
                                                    ConnectionPool.this.masterSlaveEntry.slaveUp(entry.getClient().getAddr().getHostName(), entry.getClient().getAddr().getPort(), ClientConnectionsEntry.FreezeReason.RECONNECT);
                                                    ConnectionPool.this.log.info("slave {} successfully reconnected", (Object)entry.getClient().getAddr());
                                                } else {
                                                    ClientConnectionsEntry clientConnectionsEntry = entry;
                                                    synchronized (clientConnectionsEntry) {
                                                        if (entry.getFreezeReason() == ClientConnectionsEntry.FreezeReason.RECONNECT) {
                                                            entry.setFreezed(false);
                                                            entry.setFreezeReason(null);
                                                            ConnectionPool.this.log.info("host {} successfully reconnected", (Object)entry.getClient().getAddr());
                                                        }
                                                    }
                                                }
                                            }
                                        });
                                        ConnectionPool.this.initConnections(entry, promise, false);
                                    } else {
                                        ConnectionPool.this.scheduleCheck(entry);
                                    }
                                }
                                finally {
                                    c.closeAsync();
                                }
                            }
                        };
                        if (entry.getConfig().getPassword() != null) {
                            RFuture temp = c.async(RedisCommands.AUTH, ConnectionPool.this.config.getPassword());
                            FutureListener<Void> listener = new FutureListener<Void>(){

                                @Override
                                public void operationComplete(Future<Void> future) throws Exception {
                                    ConnectionPool.this.ping(c, pingListener);
                                }
                            };
                            temp.addListener(listener);
                        } else {
                            ConnectionPool.this.ping(c, pingListener);
                        }
                    }
                });
            }
        }, this.config.getReconnectionTimeout(), TimeUnit.MILLISECONDS);
    }

    private void ping(RedisConnection c, FutureListener<String> pingListener) {
        RFuture f = c.async(RedisCommands.PING, new Object[0]);
        f.addListener(pingListener);
    }

    public void returnConnection(ClientConnectionsEntry entry, T connection) {
        if (entry.isFreezed()) {
            ((RedisConnection)connection).closeAsync();
        } else {
            this.releaseConnection(entry, connection);
        }
        this.releaseConnection(entry);
    }

    protected void releaseConnection(ClientConnectionsEntry entry) {
        entry.releaseConnection();
    }

    protected void releaseConnection(ClientConnectionsEntry entry, T conn) {
        entry.releaseConnection((RedisConnection)conn);
    }

    public static abstract class AcquireCallback<T>
    implements Runnable,
    FutureListener<T> {
    }
}

