<!--- Copyright (C) from 2022 The Play Framework Contributors <https://github.com/playframework>, 2011-2021 Lightbend Inc. <https://www.lightbend.com> -->

# Handling file upload

## Uploading files in a form using `multipart/form-data`

The standard way to upload files in a web application is to use a form with a special `multipart/form-data` encoding, which lets you mix standard form data with file attachment data.

> **Note:** The HTTP method used to submit the form must be `POST` (not `GET`).

Start by writing an HTML form:

@[file-upload-form](code/javaguide/fileupload/views/uploadForm.scala.html)

Now define the `upload` action:

@[syncUpload](code/javaguide/fileupload/controllers/HomeController.java)

The [`getRef()`](api/java/play/mvc/Http.MultipartFormData.FilePart.html#getRef\(\)) method gives you a reference to a [`TemporaryFile`](api/java/play/libs/Files.TemporaryFile.html). This is the default way Play handles file uploads.

And finally, add a `POST` route:

@[application-upload-routes](code/javaguide.upload.fileupload.routes)

> **Note:** An empty file will be treated just like no file was uploaded at all. The same applies if the `filename` header of a `multipart/form-data` file upload part is empty - even when the file itself would not empty.

### Testing the file upload

You can also write an automated JUnit test to your `upload` action:

@[testSyncUpload](code/JavaFileUploadTest.java)

Basically, we are creating a `Http.MultipartFormData.FilePart` that is required by `RequestBuilder` method `bodyMultipart`. Besides that, everything else is just like [[unit testing controllers|JavaTest#Unit-testing-controllers]].

## Direct file upload

Another way to send files to the server is to use Ajax to upload files asynchronously from a form. In this case, the request body will not be encoded as `multipart/form-data`, but will just contain the plain file contents.

@[asyncUpload](code/JavaFileUpload.java)

## Writing a custom multipart file part body parser

The multipart upload specified by [`MultipartFormData`](api/java/play/mvc/BodyParser.MultipartFormData.html) takes uploaded data from the request and puts into a TemporaryFile object.  It is possible to override this behavior so that `Multipart.FileInfo` information is streamed to another class, using the `DelegatingMultipartFormDataBodyParser` class:

@[customfileparthandler](code/JavaFileUpload.java)

Here, `pekko.stream.javadsl.FileIO` class is used to create a sink that sends the `ByteString` from the Accumulator into a `java.io.File` object, rather than a TemporaryFile object.
 
Using a custom file part handler also means that behavior can be injected, so a running count of uploaded bytes can be sent elsewhere in the system.


## Cleaning up temporary files

Uploading files uses a [`TemporaryFile`](api/java/play/libs/Files.TemporaryFile.html) API which relies on storing files in a temporary filesystem, accessible through the [`getRef()`](api/java/play/mvc/Http.MultipartFormData.FilePart.html#getRef\(\)) method.  All [`TemporaryFile`](api/java/play/libs/Files.TemporaryFile.html) references come from a [`TemporaryFileCreator`](api/java/play/libs/Files.TemporaryFileCreator.html) trait, and the implementation can be swapped out as necessary, and there's now an [`atomicMoveWithFallback`](api/java/play/libs/Files.TemporaryFile.html#atomicMoveWithFallback\(java.io.File\)) method that uses `StandardCopyOption.ATOMIC_MOVE` if available.

Uploading files is an inherently dangerous operation, because unbounded file upload can cause the filesystem to fill up -- as such, the idea behind [`TemporaryFile`](api/java/play/libs/Files.TemporaryFile.html) is that it's only in scope at completion and should be moved out of the temporary file system as soon as possible.  Any temporary files that are not moved are deleted. 

However, under [certain conditions](https://github.com/playframework/playframework/issues/5545), garbage collection does not occur in a timely fashion.  As such, there's also a [`play.api.libs.Files.TemporaryFileReaper`](api/scala/play/api/libs/Files$$DefaultTemporaryFileReaper.html) that can be enabled to delete temporary files on a scheduled basis using the Pekko scheduler, distinct from the garbage collection method.

The reaper is disabled by default, and is enabled through configuration of `application.conf`:

```
play.temporaryFile {
  reaper {
    enabled = true
    initialDelay = "5 minutes"
    interval = "30 seconds"
    olderThan = "30 minutes"
  }
}
```

The above configuration will delete files that are more than 30 minutes old, using the "olderThan" property.  It will start the reaper five minutes after the application starts, and will check the filesystem every 30 seconds thereafter.  The reaper is not aware of any existing file uploads, so protracted file uploads may run into the reaper if the system is not carefully configured.
