/*
 * Copyright (c) 2008-2011, Piccolo2D project, http://piccolo2d.org
 * Copyright (c) 1998-2008, University of Maryland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided
 * that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this list of conditions
 * and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 * and the following disclaimer in the documentation and/or other materials provided with the
 * distribution.
 *
 * None of the name of the University of Maryland, the name of the Piccolo2D project, or the names of its
 * contributors may be used to endorse or promote products derived from this software without specific
 * prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package edu.umd.cs.piccolox;

import javax.swing.JApplet;
import javax.swing.SwingUtilities;

import edu.umd.cs.piccolo.PCanvas;

/**
 * <b>PApplet</b> is meant to be subclassed by applications that just need a
 * PCanvas embedded in a web page.
 * 
 * @version 1.0
 * @author Jesse Grosjean
 */
public class PApplet extends JApplet {
    /** Used to allow versioned binary streams for serializations. */
    private static final long serialVersionUID = 1L;

    /** Canvas being displayed by this applet. */
    private PCanvas canvas;

    /**
     * Initializes the applet with a canvas and no background.
     */
    public void init() {
        setBackground(null);

        canvas = createCanvas();
        getContentPane().add(canvas);
        validate();
        canvas.requestFocus();
        beforeInitialize();

        // Manipulation of Piccolo's scene graph should be done from Swings
        // event dispatch thread since Piccolo is not thread safe. This code
        // calls initialize() from that thread once the PFrame is initialized,
        // so you are safe to start working with Piccolo in the initialize()
        // method.
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                PApplet.this.initialize();
                repaint();
            }
        });
    }

    /**
     * Returns the canvas this PApplet is displaying.
     * 
     * @return canvas this applet is displaying
     */
    public PCanvas getCanvas() {
        return canvas;
    }

    /**
     * Provides an extension point for subclasses so that they can control
     * what's on the canvas by default.
     * 
     * @return a built canvas
     */
    public PCanvas createCanvas() {
        return new PCanvas();
    }

    /**
     * This method will be called before the initialize() method and will be
     * called on the thread that is constructing this object.
     */
    public void beforeInitialize() {
    }

    /**
     * Subclasses should override this method and add their Piccolo2d
     * initialization code there. This method will be called on the swing event
     * dispatch thread. Note that the constructors of PFrame subclasses may not
     * be complete when this method is called. If you need to initailize some
     * things in your class before this method is called place that code in
     * beforeInitialize();
     */
    public void initialize() {
    }
}