// SPDX-FileCopyrightText: 2025 Paul Schaub <vanitasvitae@fsfe.org>
//
// SPDX-License-Identifier: Apache-2.0

package org.pgpainless.algorithm

enum class OpenPGPKeyVersion(val numeric: Int) {
    // Version 2 packets are identical in format to Version 3 packets, but are generated by
    // PGP 2.5 or before. V2 packets are deprecated and they MUST NOT be generated.

    /**
     * Version 3 packets were first generated by PGP 2.6. Version 3 keys are deprecated. They
     * contain three weaknesses. First, it is relatively easy to construct a version 3 key that has
     * the same Key ID as any other key because the Key ID is simply the low 64 bits of the public
     * modulus. Second, because the fingerprint of a version 3 key hashes the key material, but not
     * its length, there is an increased opportunity for fingerprint collisions. Third, there are
     * weaknesses in the MD5 hash algorithm that cause developers to prefer other algorithms.
     */
    @Deprecated("V3 keys are deprecated.") v3(3),

    /**
     * Version 4 packets are used in RFC2440, RFC4880, RFC9580. The version 4 format is widely
     * supported by various implementations.
     *
     * @see [RFC2440](https://www.rfc-editor.org/rfc/rfc2440.html)
     * @see [RFC4880](https://www.rfc-editor.org/rfc/rfc4880.html)
     * @see [RFC9580](https://www.rfc-editor.org/rfc/rfc9580.html)
     */
    v4(4),

    /**
     * "V5"-keys are introduced in the LibrePGP document. These are NOT OpenPGP keys and are
     * primarily supported by GnuPG and RNP.
     *
     * @see [LibrePGP](https://datatracker.ietf.org/doc/draft-koch-librepgp/)
     */
    librePgp(5),

    /**
     * Version 6 packets are introduced in RFC9580. The version 6 format is similar to the version 4
     * format except for the addition of a count for the key material. This count helps parsing
     * Secret Key packets (which are an extension of the Public Key packet format) in the case of an
     * unknown algorithm. In addition, fingerprints of version 6 keys are calculated differently
     * from version 4 keys, preventing the KOpenPGP attack.
     *
     * @see [RFC9580](https://www.rfc-editor.org/rfc/rfc9580.html)
     */
    v6(6),
    ;

    companion object {
        @JvmStatic
        fun from(numeric: Int): OpenPGPKeyVersion {
            return values().find { numeric == it.numeric }
                ?: throw IllegalArgumentException("Unknown key version $numeric")
        }
    }
}
