// SPDX-FileCopyrightText: 2023 Paul Schaub <vanitasvitae@fsfe.org>
//
// SPDX-License-Identifier: Apache-2.0

package org.pgpainless

import java.io.ByteArrayOutputStream
import java.io.OutputStream
import java.util.*
import org.bouncycastle.bcpg.ArmoredOutputStream
import org.bouncycastle.bcpg.BCPGOutputStream
import org.bouncycastle.bcpg.PacketFormat
import org.bouncycastle.openpgp.PGPKeyRing
import org.bouncycastle.openpgp.PGPPublicKeyRing
import org.bouncycastle.openpgp.PGPSecretKeyRing
import org.bouncycastle.openpgp.PGPSignature
import org.bouncycastle.openpgp.api.OpenPGPApi
import org.bouncycastle.openpgp.api.OpenPGPCertificate
import org.bouncycastle.openpgp.api.OpenPGPImplementation
import org.bouncycastle.openpgp.api.OpenPGPKey
import org.bouncycastle.openpgp.api.OpenPGPKeyGenerator
import org.bouncycastle.openpgp.api.OpenPGPKeyReader
import org.bouncycastle.openpgp.api.OpenPGPSignature
import org.bouncycastle.openpgp.api.bc.BcOpenPGPApi
import org.pgpainless.algorithm.OpenPGPKeyVersion
import org.pgpainless.bouncycastle.PolicyAdapter
import org.pgpainless.bouncycastle.extensions.setAlgorithmSuite
import org.pgpainless.decryption_verification.DecryptionBuilder
import org.pgpainless.encryption_signing.EncryptionBuilder
import org.pgpainless.key.certification.CertifyCertificate
import org.pgpainless.key.generation.KeyRingBuilder
import org.pgpainless.key.generation.KeyRingTemplates
import org.pgpainless.key.info.KeyRingInfo
import org.pgpainless.key.modification.secretkeyring.SecretKeyRingEditor
import org.pgpainless.key.parsing.KeyRingReader
import org.pgpainless.policy.Policy
import org.pgpainless.util.ArmorUtils

/**
 * Main entry point to the PGPainless OpenPGP API. Historically, this class was used through static
 * factory methods only, and configuration was done using the Singleton pattern. However, now it is
 * recommended to instantiate the API and apply configuration on a per-instance manner. The benefit
 * of this being that you can have multiple differently configured instances at the same time.
 *
 * @param implementation OpenPGP Implementation - either BCs lightweight
 *   [org.bouncycastle.openpgp.api.bc.BcOpenPGPImplementation] or JCAs
 *   [org.bouncycastle.openpgp.api.jcajce.JcaOpenPGPImplementation].
 * @param algorithmPolicy policy, deciding acceptable algorithms
 */
class PGPainless(
    val implementation: OpenPGPImplementation = OpenPGPImplementation.getInstance(),
    val algorithmPolicy: Policy = Policy()
) {

    constructor(
        algorithmPolicy: Policy
    ) : this(OpenPGPImplementation.getInstance(), algorithmPolicy)

    private val api: OpenPGPApi

    init {
        implementation.setPolicy(
            PolicyAdapter(algorithmPolicy)) // adapt PGPainless' Policy to BCs OpenPGPPolicy
        api = BcOpenPGPApi(implementation)
    }

    @JvmOverloads
    fun toAsciiArmor(
        certOrKey: OpenPGPCertificate,
        packetFormat: PacketFormat = PacketFormat.ROUNDTRIP
    ): String {
        val armorBuilder = ArmoredOutputStream.builder().clearHeaders()
        ArmorUtils.keyToHeader(certOrKey.primaryKey.pgpPublicKey)
            .getOrDefault(ArmorUtils.HEADER_COMMENT, setOf())
            .forEach { armorBuilder.addComment(it) }
        return certOrKey.toAsciiArmoredString(packetFormat, armorBuilder)
    }

    @JvmOverloads
    fun toAsciiArmor(
        certsOrKeys: Collection<OpenPGPCertificate>,
        packetFormat: PacketFormat = PacketFormat.ROUNDTRIP
    ): String {
        return certsOrKeys.joinToString(
            separator = "\n", transform = { it.toAsciiArmoredString(packetFormat) })
    }

    @JvmOverloads
    fun toAsciiArmor(
        signature: OpenPGPSignature,
        packetFormat: PacketFormat = PacketFormat.ROUNDTRIP
    ): String {
        val armorBuilder = ArmoredOutputStream.builder().clearHeaders()
        armorBuilder.addComment(signature.keyIdentifier.toPrettyPrint())
        return signature.toAsciiArmoredString(packetFormat, armorBuilder)
    }

    @JvmOverloads
    fun toAsciiArmor(
        signature: PGPSignature,
        packetFormat: PacketFormat = PacketFormat.ROUNDTRIP
    ): String {
        val armorBuilder = ArmoredOutputStream.builder().clearHeaders()
        OpenPGPSignature.getMostExpressiveIdentifier(signature.keyIdentifiers)?.let {
            armorBuilder.addComment(it.toPrettyPrint())
        }
        val bOut = ByteArrayOutputStream()
        val aOut = armorBuilder.build(bOut)
        val pOut = BCPGOutputStream(aOut, packetFormat)
        signature.encode(pOut)
        pOut.close()
        aOut.close()
        return bOut.toString()
    }

    /**
     * Generate a new [OpenPGPKey] from predefined templates.
     *
     * @param version [OpenPGPKeyVersion], defaults to [OpenPGPKeyVersion.v4]
     * @param creationTime of the key, defaults to now
     * @return [KeyRingTemplates] api
     */
    @JvmOverloads
    fun generateKey(
        version: OpenPGPKeyVersion = OpenPGPKeyVersion.v4,
        creationTime: Date = Date()
    ): KeyRingTemplates = KeyRingTemplates(version, creationTime, this)

    /**
     * Build a fresh, custom [OpenPGPKey] using PGPainless' API.
     *
     * @param version [OpenPGPKeyVersion], defaults to [OpenPGPKeyVersion.v4]
     * @return [KeyRingBuilder] api
     */
    @JvmOverloads
    fun buildKey(version: OpenPGPKeyVersion = OpenPGPKeyVersion.v4): KeyRingBuilder =
        KeyRingBuilder(version, this)

    /**
     * Build a fresh, custom [OpenPGPKey] using BCs new API.
     *
     * @param version [OpenPGPKeyVersion], defaults to [OpenPGPKeyVersion.v4]
     * @param creationTime creation time of the key, defaults to now
     * @return [OpenPGPKeyGenerator] api
     */
    @JvmOverloads
    fun _buildKey(
        version: OpenPGPKeyVersion = OpenPGPKeyVersion.v4,
        creationTime: Date = Date()
    ): OpenPGPKeyGenerator =
        OpenPGPKeyGenerator(
                implementation,
                version.numeric,
                algorithmPolicy.keyProtectionSettings.aead,
                creationTime)
            .setAlgorithmSuite(algorithmPolicy.keyGenerationAlgorithmSuite)

    /**
     * Inspect an [OpenPGPKey] or [OpenPGPCertificate], gaining convenient access to its properties.
     *
     * @param keyOrCertificate [OpenPGPKey] or [OpenPGPCertificate]
     * @param referenceTime reference time for evaluation
     * @return [KeyRingInfo] wrapper
     */
    @JvmOverloads
    fun inspect(keyOrCertificate: OpenPGPCertificate, referenceTime: Date = Date()): KeyRingInfo =
        KeyRingInfo(keyOrCertificate, this, referenceTime)

    /**
     * Modify an [OpenPGPKey], adding new components and signatures. This API can be used to add new
     * subkeys, user-ids or user-attributes to the key, extend or alter its expiration time, revoke
     * individual components of the entire certificate, etc.
     *
     * @param key key to modify
     * @param referenceTime timestamp for modifications
     * @return [SecretKeyRingEditor] api
     */
    @JvmOverloads
    fun modify(key: OpenPGPKey, referenceTime: Date = Date()): SecretKeyRingEditor =
        SecretKeyRingEditor(key, this, referenceTime)

    /**
     * Parse [OpenPGPKey]/[OpenPGPCertificate] material from binary or ASCII armored encoding.
     *
     * @return [OpenPGPKeyReader] api
     */
    fun readKey(): OpenPGPKeyReader = api.readKeyOrCertificate()

    /**
     * Convert a [PGPSecretKeyRing] into an [OpenPGPKey].
     *
     * @param secretKeyRing mid-level API [PGPSecretKeyRing] object
     * @return high-level API [OpenPGPKey] object
     */
    fun toKey(secretKeyRing: PGPSecretKeyRing): OpenPGPKey =
        OpenPGPKey(secretKeyRing, implementation)

    /**
     * Convert a [PGPPublicKeyRing] into an [OpenPGPCertificate].
     *
     * @param certificate mid-level API [PGPSecretKeyRing] object
     * @return high-level API [OpenPGPCertificate] object
     */
    fun toCertificate(certificate: PGPPublicKeyRing): OpenPGPCertificate =
        OpenPGPCertificate(certificate, implementation)

    /**
     * Depending on the type, convert either a [PGPSecretKeyRing] into an [OpenPGPKey] or a
     * [PGPPublicKeyRing] into an [OpenPGPCertificate].
     *
     * @param keyOrCertificate [PGPKeyRing], either [PGPSecretKeyRing] or [PGPPublicKeyRing]
     * @return depending on the type of [keyOrCertificate], either an [OpenPGPKey] or
     *   [OpenPGPCertificate]
     */
    fun toKeyOrCertificate(keyOrCertificate: PGPKeyRing): OpenPGPCertificate =
        when (keyOrCertificate) {
            is PGPSecretKeyRing -> toKey(keyOrCertificate)
            is PGPPublicKeyRing -> toCertificate(keyOrCertificate)
            else ->
                throw IllegalArgumentException(
                    "Unexpected PGPKeyRing subclass: ${keyOrCertificate.javaClass.name}")
        }

    /**
     * Merge two copies of an [OpenPGPCertificate] into a single copy. This method can be used to
     * import new third-party signatures into a certificate.
     *
     * @param originalCopy local copy of the certificate
     * @param updatedCopy copy of the same certificate, potentially carrying new signatures and
     *   components
     * @return merged [OpenPGPCertificate]
     */
    fun mergeCertificate(
        originalCopy: OpenPGPCertificate,
        updatedCopy: OpenPGPCertificate
    ): OpenPGPCertificate {
        return OpenPGPCertificate.join(originalCopy, updatedCopy)
    }

    /**
     * Generate an encrypted and/or signed OpenPGP message.
     *
     * @return [EncryptionBuilder] api
     */
    fun generateMessage(): EncryptionBuilder = EncryptionBuilder(this)

    /**
     * Process an OpenPGP message. This method attempts decryption of encrypted messages and
     * performs signature verification.
     *
     * @return [DecryptionBuilder] api
     */
    fun processMessage(): DecryptionBuilder = DecryptionBuilder(this)

    /**
     * Create certification signatures on third-party [OpenPGPCertificates][OpenPGPCertificate].
     *
     * @return [CertifyCertificate] api
     */
    fun generateCertification(): CertifyCertificate = CertifyCertificate(this)

    companion object {

        @Volatile private var instance: PGPainless? = null

        @JvmStatic
        fun getInstance(): PGPainless =
            instance ?: synchronized(this) { instance ?: PGPainless().also { instance = it } }

        @JvmStatic
        fun setInstance(api: PGPainless) {
            instance = api
        }

        /**
         * Generate a fresh [OpenPGPKey] from predefined templates.
         *
         * @return templates
         */
        @JvmStatic
        @JvmOverloads
        @Deprecated(
            "Call .generateKey() on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("generateKey(version)"))
        fun generateKeyRing(version: OpenPGPKeyVersion = OpenPGPKeyVersion.v4): KeyRingTemplates =
            getInstance().generateKey(version)

        /**
         * Build a custom OpenPGP key ring.
         *
         * @return builder
         */
        @JvmStatic
        @JvmOverloads
        @Deprecated(
            "Call buildKey() on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("buildKey(version)"))
        fun buildKeyRing(version: OpenPGPKeyVersion = OpenPGPKeyVersion.v4): KeyRingBuilder =
            getInstance().buildKey(version)

        /**
         * Read an existing OpenPGP key ring.
         *
         * @return builder
         */
        @JvmStatic
        @Deprecated("Use readKey() instead.", replaceWith = ReplaceWith("readKey()"))
        fun readKeyRing(): KeyRingReader = KeyRingReader()

        /**
         * Extract a public key certificate from a secret key.
         *
         * @param secretKey secret key
         * @return public key certificate
         */
        @JvmStatic
        @Deprecated("Use .toKey() and then .toCertificate() instead.")
        fun extractCertificate(secretKey: PGPSecretKeyRing): PGPPublicKeyRing =
            secretKey.toCertificate()

        /**
         * Merge two copies of the same certificate (e.g. an old copy, and one retrieved from a key
         * server) together.
         *
         * @param originalCopy local, older copy of the cert
         * @param updatedCopy updated, newer copy of the cert
         * @return merged certificate
         * @throws PGPException in case of an error
         */
        @JvmStatic
        @Deprecated(
            "Use mergeCertificate() instead.", replaceWith = ReplaceWith("mergeCertificate()"))
        fun mergeCertificate(
            originalCopy: PGPPublicKeyRing,
            updatedCopy: PGPPublicKeyRing
        ): PGPPublicKeyRing = PGPPublicKeyRing.join(originalCopy, updatedCopy)

        /**
         * Wrap a key or certificate in ASCII armor.
         *
         * @param key key or certificate
         * @return ascii armored string
         * @throws IOException in case of an error during the armoring process
         */
        @JvmStatic
        fun asciiArmor(key: PGPKeyRing): String =
            getInstance().toAsciiArmor(getInstance().toKeyOrCertificate(key))

        @JvmStatic
        @Deprecated(
            "Call getInstance().toAsciiArmor(cert) instead.",
            replaceWith = ReplaceWith("getInstance().toAsciiArmor(cert)"))
        fun asciiArmor(cert: OpenPGPCertificate): String = getInstance().toAsciiArmor(cert)

        /**
         * Wrap a key of certificate in ASCII armor and write the result into the given
         * [OutputStream].
         *
         * @param key key or certificate
         * @param outputStream output stream
         * @throws IOException in case of an error during the armoring process
         */
        @JvmStatic
        fun asciiArmor(key: PGPKeyRing, outputStream: OutputStream) {
            val armorOut = ArmorUtils.toAsciiArmoredStream(key, outputStream)
            key.encode(armorOut)
            armorOut.close()
        }

        /**
         * Wrap the detached signature in ASCII armor.
         *
         * @param signature detached signature
         * @return ascii armored string
         * @throws IOException in case of an error during the armoring process
         */
        @JvmStatic
        @Deprecated(
            "Call toAsciiArmor(signature) on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("getInstance().toAsciiArmor(signature)"))
        fun asciiArmor(signature: PGPSignature): String = getInstance().toAsciiArmor(signature)

        /**
         * Create an [EncryptionBuilder], which can be used to encrypt and/or sign data using
         * OpenPGP.
         *
         * @return builder
         */
        @Deprecated(
            "Call generateMessage() on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("generateMessage()"))
        @JvmStatic
        fun encryptAndOrSign(): EncryptionBuilder = getInstance().generateMessage()

        /**
         * Create a [DecryptionBuilder], which can be used to decrypt and/or verify data using
         * OpenPGP.
         *
         * @return builder
         */
        @Deprecated(
            "Call processMessage() on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("processMessage()"))
        @JvmStatic
        fun decryptAndOrVerify(): DecryptionBuilder = getInstance().processMessage()

        /**
         * Make changes to a secret key at the given reference time. This method can be used to
         * change key expiration dates and passphrases, or add/revoke user-ids and subkeys.
         *
         * <p>
         * After making the desired changes in the builder, the modified key can be extracted using
         * [org.pgpainless.key.modification.secretkeyring.SecretKeyRingEditorInterface.done].
         *
         * @param secretKeys secret key ring
         * @param referenceTime reference time used as signature creation date
         * @return builder
         */
        @JvmStatic
        @JvmOverloads
        fun modifyKeyRing(
            secretKey: PGPSecretKeyRing,
            referenceTime: Date = Date()
        ): SecretKeyRingEditor = getInstance().modify(getInstance().toKey(secretKey), referenceTime)

        /**
         * Quickly access information about a [org.bouncycastle.openpgp.PGPPublicKeyRing] /
         * [PGPSecretKeyRing]. This method can be used to determine expiration dates, key flags and
         * other information about a key at a specific time.
         *
         * @param key key ring
         * @param referenceTime date of inspection
         * @return access object
         */
        @JvmStatic
        @JvmOverloads
        @Deprecated(
            "Use inspect(key) on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("inspect(key)"))
        fun inspectKeyRing(key: PGPKeyRing, referenceTime: Date = Date()): KeyRingInfo =
            getInstance().inspect(getInstance().toKeyOrCertificate(key), referenceTime)

        @JvmStatic
        @JvmOverloads
        @Deprecated(
            "Use inspect(key) on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("inspect(key)"))
        fun inspectKeyRing(key: OpenPGPCertificate, referenceTime: Date = Date()): KeyRingInfo =
            getInstance().inspect(key, referenceTime)

        /**
         * Access, and make changes to PGPainless policy on acceptable/default algorithms etc.
         *
         * @return policy
         */
        @Deprecated(
            "Use PGPainless.getInstance().getAlgorithmPolicy() instead.",
            replaceWith = ReplaceWith("getInstance().algorithmPolicy"))
        @JvmStatic
        fun getPolicy(): Policy = getInstance().algorithmPolicy

        /**
         * Create different kinds of signatures on other keys.
         *
         * @return builder
         */
        @Deprecated(
            "Call .generateCertification() on an instance of PGPainless instead.",
            replaceWith = ReplaceWith("generateCertification()"))
        @JvmStatic
        fun certify(): CertifyCertificate = getInstance().generateCertification()
    }
}
