// ===========================================================================
// CONTENT  : CLASS SimpleJsonParser
// AUTHOR   : Manfred Duchrow
// VERSION  : 1.1 - 14/03/2020
// HISTORY  :
//  24/08/2014  mdu  CREATED
//  14/03/2020  mdu   added -> create()
//
// Copyright (c) 2014-2020, by MDCS. All rights reserved.
// ===========================================================================
package org.pfsw.text.json;

import static org.pfsw.text.json.JsonConstants.*;

import java.math.BigDecimal;
import java.util.Map;

import org.pfsw.text.StringExaminer;
import org.pfsw.text.StringUtil;

/**
 * A parser that creates a Java internal structure of JsonObject and
 * JsonArray objects from a given JSON text.
 * <p>
 * The data will be mapped on the following Java types:
 * <dl>
 * <dt>Object</dt><dd>JsonObject</dd>
 * <dt>Array</dt><dd>JsonArray</dd>
 * <dt>null</dt><dd>null</dd>
 * <dt>true</dt><dd>Boolean.TRUE</dd>
 * <dt>false</dt><dd>Boolean.FALSE</dd>
 * <dt>"string"</dt><dd>String</dd>
 * <dt>integer number</dt><dd>Integer</dd>
 * <dt>long number</dt><dd>Long</dd>
 * <dt>other numbers</dt><dd>BigDecimal</dd>
 * </dl>
 * <p>
 * It is based on the RFC-7159 (http://tools.ietf.org/html/rfc7159).
 *
 * @author Manfred Duchrow
 * @version 1.1
 */
public class SimpleJsonParser
{
  // =========================================================================
  // CONSTANTS
  // =========================================================================
  private static final StringUtil SU = StringUtil.current();

  // =========================================================================
  // INSTANCE VARIABLES
  // =========================================================================
  private final CharEscaper charEscaper = new CharEscaper();
  private StringExaminer stringExaminer;

  // =========================================================================
  // CLASS METHODS
  // =========================================================================
  /**
   * Creates a new parser.
   */
  public static SimpleJsonParser create()
  {
    return new SimpleJsonParser();
  }
  
  // =========================================================================
  // CONSTRUCTORS
  // =========================================================================
  /**
   * Creates a new parser.
   */
  public SimpleJsonParser()
  {
    super();
  }

  // =========================================================================
  // PUBLIC INSTANCE METHODS
  // =========================================================================
  /**
   * Parse the given JSON text and return its Java representation. 
   * That is one of the following types: 
   * <ul>
   * <li>JsonObject</li>
   * <li>JasonArray</li>
   * <li>String</li>
   * <li>Boolean</li>
   * <li>Integer</li>
   * <li>Long</li>
   * <li>BigDecimal</li>
   * <li>null</li>
   * </ul> 
   * 
   * @param jsonText The JSON text to be parsed (must not be null).
   * @return The parsed data as Java objects.
   * @throws JsonParseException Any parsing problem.
   */
  public Object parse(final String jsonText) throws JsonParseException
  {
    setStringExaminer(new StringExaminer(jsonText));
    return parseInternal();
  }

  /**
   * Parse the given JSON text and return its Java representation as JsonObject. 
   * That implies that the given JSON text starts with a "{".
   * 
   * @param jsonText The JSON text to be parsed (must not be null).
   * @return The parsed data as Java objects.
   * @throws JsonParseException Any parsing problem.
   * @throws ClassCastException If the JSON text did not represent a JSON object.
   */
  public JsonObject parseObject(String jsonText) throws JsonParseException
  {
    return parseTo(jsonText, JsonObject.class);
  }

  /**
   * Parse the given JSON text and return its Java representation as JsonArray. 
   * That implies that the given JSON text starts with a "[".
   * 
   * @param jsonText The JSON text to be parsed (must not be null).
   * @return The parsed data as Java objects.
   * @throws JsonParseException Any parsing problem.
   * @throws ClassCastException If the JSON text did not represent a JSON array.
   */
  public JsonArray parseArray(String jsonText) throws JsonParseException
  {
    return parseTo(jsonText, JsonArray.class);
  }

  // =========================================================================
  // PROTECTED INSTANCE METHODS
  // =========================================================================
  protected boolean isObjectStart(final char ch)
  {
    return (BEGIN_OBJECT == ch);
  }

  protected boolean isObjectEnd(final char ch)
  {
    return (END_OBJECT == ch);
  }

  protected boolean isArrayStart(final char ch)
  {
    return (BEGIN_ARRAY == ch);
  }

  protected boolean isArrayEnd(final char ch)
  {
    return (END_ARRAY == ch);
  }

  protected boolean isStringStart(final char ch)
  {
    return (getStringDelimiter() == ch);
  }

  protected boolean isStringEnd(final char ch)
  {
    return (getStringDelimiter() == ch);
  }

  protected boolean isEscape(final char ch)
  {
    return (ESCAPE == ch);
  }

  protected boolean isValueSeparator(final char ch)
  {
    return (VALUE_SEPARATOR == ch);
  }

  protected boolean isNumberStart(final char ch)
  {
    return Character.isDigit(ch) || (ch == '-') || (ch == '+');
  }

  protected Object parseInternal() throws JsonParseException
  {
    // According to RFC-7159 all values are allowed, not only Object and Array
    return parseValue();
  }

  protected Object parseValue() throws JsonParseException
  {
    char ch;

    ch = peekNextNoneWhitespaceChar();

    if (isObjectStart(ch))
    {
      return parseObject();
    }
    if (isArrayStart(ch))
    {
      return parseArray();
    }
    if (isStringStart(ch))
    {
      return parseString();
    }
    if (isNumberStart(ch))
    {
      return parseNumber();
    }
    return parseLiteral();
  }

  protected JsonObject parseObject() throws JsonParseException
  {
    JsonObject jsonObject;
    char ch;

    checkNextMandatoryToken(BEGIN_OBJECT);
    jsonObject = createObject();

    ch = peekNextNoneWhitespaceChar();
    if (isObjectEnd(ch))
    {
      nextChar(); // Read it away
      return jsonObject;
    }

    while (true)
    {
      parseKeyValueInto(jsonObject);

      ch = nextNoneWhitespaceChar();
      if (isObjectEnd(ch))
      {
        return jsonObject;
      }
      if (!isValueSeparator(ch))
      {
        throw new JsonParseException("Value separator '" + VALUE_SEPARATOR + "' expected at position: " + getLastPosition());
      }
    }
  }

  protected void parseKeyValueInto(final Map<String, Object> jsonObject) throws JsonParseException
  {
    String key;
    Object value;

    key = parseString();
    checkNextMandatoryToken(NAME_SEPARATOR);
    value = parseValue();
    jsonObject.put(key, value);
  }

  protected JsonArray parseArray() throws JsonParseException
  {
    JsonArray jsonArray;
    char ch;
    Object value;

    checkNextMandatoryToken(BEGIN_ARRAY);
    jsonArray = createArray();

    ch = peekNextNoneWhitespaceChar();
    if (isArrayEnd(ch))
    {
      nextChar(); // Read it away
      return jsonArray;
    }

    while (true)
    {
      value = parseValue();
      jsonArray.add(value);

      ch = nextNoneWhitespaceChar();
      if (isArrayEnd(ch))
      {
        return jsonArray;
      }
      if (!isValueSeparator(ch))
      {
        throw new JsonParseException("Value separator '" + VALUE_SEPARATOR + "' expected in array at position: " + getLastPosition());
      }
    }
  }

  protected String parseString() throws JsonParseException
  {
    String string;

    checkNextMandatoryToken(getStringDelimiter());
    string = parseUpToQuote();
    return string;
  }

  protected String parseUpToQuote() throws JsonParseException
  {
    StringBuilder buffer;
    char ch;

    buffer = new StringBuilder(200);
    ch = nextChar();
    while (!isStringEnd(ch))
    {
      if (isEscape(ch))
      {
        ch = nextChar();
        ch = getCharEscaper().getCharForEscapedChar(ch);
      }
      buffer.append(ch);
      ch = nextChar();
    }
    return buffer.toString();
  }

  /**
   * Returns either an Integer, a Long or a BigDecimal.
  
   * @throws JsonParseException If the string cannot be parsed to a number.
   */
  protected Number parseNumber() throws JsonParseException
  {
    String string;
    BigDecimal number;
    int pos;

    pos = getLastPosition();
    string = parseLiteralString(30, "+-0123456789.eE");
    try
    {
      number = new BigDecimal(string);
    }
    catch (@SuppressWarnings("unused") NumberFormatException ex)
    {
      throw new JsonParseException("Invalid number: " + string + " , at position: " + pos);
    }
    if (number.scale() == 0)
    {
      if (number.compareTo(BigDecimal.valueOf(Integer.MAX_VALUE)) < 0)
      {
        return Integer.valueOf(number.intValue());
      }
      return Long.valueOf(number.longValue());
    }
    return number;
  }

  protected String parseLiteralString(int maxLength, String allowedChars) throws JsonParseException
  {
    StringBuilder buffer;
    char ch;

    buffer = new StringBuilder(maxLength);
    ch = peekNextChar();
    while ((buffer.length() < maxLength) && (SU.contains(allowedChars, ch)))
    {
      buffer.append(nextChar());
      ch = peekNextChar();
    }

    return buffer.toString();
  }

  protected Object parseLiteral() throws JsonParseException
  {
    String text;
    int pos;

    peekNextNoneWhitespaceChar();
    pos = getLastPosition();
    text = parseLiteralString(5, "truefalsn");
    if (TRUE.equals(text))
    {
      return Boolean.TRUE;
    }
    if (FALSE.equals(text))
    {
      return Boolean.FALSE;
    }
    if (NULL.equals(text))
    {
      return null;
    }
    throw new JsonParseException("No valid JSON literal: " + text + "   at position " + pos);
  }

  @SuppressWarnings("unchecked")
  protected <T extends JsonType> T parseTo(final String json, final Class<T> resultType) throws JsonParseException
  {
    Object result;

    result = parse(json);
    if (resultType.isInstance(result))
    {
      return (T)result;
    }
    throw new JsonParseException("Class cast problem for parsed JSON. Actual class is: " + result.getClass().getName());
  }

  protected int getLastPosition()
  {
    return getStringExaminer().getPosition();
  }

  protected void checkNextMandatoryToken(char expectedChar) throws JsonParseException
  {
    char ch;

    ch = nextNoneWhitespaceChar();
    if (ch != expectedChar)
    {
      throw new JsonParseException("Expected token <" + expectedChar + "> not found at position " + getLastPosition());
    }
  }

  /**
   * Looks for the next character that is no white space and returns it.
   * Position when finished: on next character after the one returned. 
   * @throws JsonParseException If the end of the underlying JSON data was reached.
   */
  protected char nextNoneWhitespaceChar() throws JsonParseException
  {
    char ch;

    ch = getStringExaminer().nextNoneWhitespaceChar();
    if (getStringExaminer().endReached(ch))
    {
      throw new JsonParseException("End of data reached unexpectedly.");
    }
    return ch;
  }

  /**
   * Returns simply the next character of the underlying JSON string.
   * Position when finished: on next character after the one returned. 
   * @throws JsonParseException If the end of the underlying JSON data was reached.
   */
  protected char nextChar() throws JsonParseException
  {
    char ch;

    ch = getStringExaminer().nextChar();
    if (getStringExaminer().endReached(ch))
    {
      throw new JsonParseException("Unexpected end of data reached.");
    }
    return ch;
  }

  /**
   * Returns the next character of the underlying JSON string without moving the 
   * examiner's position forward. That is, the next nextChar() call will return 
   * the same character again.
   * Position when finished: still on same character. 
   */
  protected char peekNextChar()
  {
    return getStringExaminer().peek();
  }

  /**
   * Looks for the next character that is no white space and returns it.
   * Position when finished: on the character returned. 
   * @throws JsonParseException If the end of the underlying JSON data was reached.
   */
  protected char peekNextNoneWhitespaceChar() throws JsonParseException
  {
    char ch;

    ch = nextNoneWhitespaceChar();
    getStringExaminer().skip(-1);
    return ch;
  }

  protected char getStringDelimiter()
  {
    return STRING_DELIMITER;
  }

  protected JsonObject createObject()
  {
    return new JsonObject();
  }

  protected JsonArray createArray()
  {
    return new JsonArray();
  }

  protected StringExaminer getStringExaminer()
  {
    return this.stringExaminer;
  }

  protected void setStringExaminer(StringExaminer stringExaminer)
  {
    this.stringExaminer = stringExaminer;
  }

  public CharEscaper getCharEscaper()
  {
    return this.charEscaper;
  }

  /**
   * Just for debugging purposes!
   */
  protected String getCurrentStringRegion(int backwards, int forwards)
  {
    int len;
    StringBuilder buffer;

    len = backwards + forwards;
    buffer = new StringBuilder(len);
    getStringExaminer().markPosition();
    getStringExaminer().skip(backwards * (-1));
    for (int i = 0; i < len; i++)
    {
      buffer.append(getStringExaminer().nextChar());
    }
    getStringExaminer().restorePosition();
    return buffer.toString();
  }
}
